/**
 * WhatsApp Utilities
 * دوال مساعدة عامة لتنظيف أرقام WhatsApp وإنشاء الروابط
 */

/**
 * تنظيف رقم WhatsApp من أي رموز أو معاملات إضافية
 * @param {string|number} number - رقم WhatsApp (قد يحتوي على رموز أو معاملات أو رابط كامل)
 * @returns {string} - رقم نظيف يحتوي فقط على الأرقام
 */
export function cleanWhatsAppNumber(number) {
  if (!number) return '';
  
  let numberStr = String(number).trim();
  
  // إذا كان الرابط كاملاً (wa.me أو api.whatsapp.com)، استخرج الرقم منه
  if (numberStr.includes('wa.me/')) {
    const match = numberStr.match(/wa\.me\/(\d+)/);
    if (match && match[1]) {
      return match[1];
    }
  }
  
  if (numberStr.includes('api.whatsapp.com/send')) {
    const match = numberStr.match(/phone=(\d+)/);
    if (match && match[1]) {
      return match[1];
    }
  }
  
  // إذا كان يحتوي على معاملات (مثل ?text=...)، استخرج الرقم قبل علامة ?
  if (numberStr.includes('?')) {
    numberStr = numberStr.split('?')[0];
  }
  
  // تنظيف الرقم من كل شيء ما عدا الأرقام
  const cleaned = numberStr.replace(/[^0-9]/g, '');
  
  // التأكد من أن الرقم ليس فارغاً وأنه يحتوي على رقم صالح (على الأقل 7 أرقام)
  if (cleaned && cleaned.length >= 7) {
    return cleaned;
  }
  
  return '';
}

/**
 * إنشاء رابط WhatsApp نظيف
 * @param {string|number} number - رقم WhatsApp
 * @param {string} message - رسالة مسبقة (اختياري)
 * @returns {string} - رابط WhatsApp نظيف
 */
export function getWhatsAppLink(number, message = null) {
  if (!number) return '#';
  
  const cleanedNumber = cleanWhatsAppNumber(number);
  if (!cleanedNumber) return '#';
  
  let url = `https://wa.me/${cleanedNumber}`;
  
  if (message) {
    const encodedMessage = encodeURIComponent(message);
    url += `?text=${encodedMessage}`;
  }
  
  return url;
}

/**
 * الحصول على رقم WhatsApp للعرض (منظف)
 * @param {string|number} number - رقم WhatsApp
 * @returns {string} - رقم نظيف للعرض
 */
export function getWhatsAppDisplay(number) {
  if (!number) return '';
  return cleanWhatsAppNumber(number);
}

/**
 * فتح رابط WhatsApp في نافذة جديدة
 * @param {string|number} number - رقم WhatsApp
 * @param {string} message - رسالة مسبقة (اختياري)
 */
export function openWhatsApp(number, message = null) {
  const link = getWhatsAppLink(number, message);
  console.log('Opening WhatsApp link:', link);
  console.log('Original number:', number);
  console.log('Cleaned number:', cleanWhatsAppNumber(number));
  
  if (link && link !== '#') {
    window.open(link, '_blank', 'noopener,noreferrer');
  } else {
    console.error('Invalid WhatsApp number:', number);
  }
}

export default {
  cleanWhatsAppNumber,
  getWhatsAppLink,
  getWhatsAppDisplay,
  openWhatsApp,
};

