/**
 * Utility functions for managing cookie storage data
 * دوال مساعدة لإدارة بيانات cookie storage
 */
'use client';

import cookieStorage from './cookieStorage';

/**
 * حفظ بيانات الحجز المؤقتة (Booking Step Data)
 * Save temporary booking step data
 * @param {object} data - بيانات الحجز
 */
export const saveBookingStepData = (data) => {
  try {
    cookieStorage.setItem('bookingStepData', JSON.stringify(data));
  } catch (error) {
    console.error('Error saving booking step data:', error);
  }
};

/**
 * استرجاع بيانات الحجز المؤقتة
 * Get temporary booking step data
 * @returns {object|null} - بيانات الحجز أو null
 */
export const getBookingStepData = () => {
  try {
    const data = cookieStorage.getItem('bookingStepData');
    return data ? JSON.parse(data) : null;
  } catch (error) {
    console.error('Error getting booking step data:', error);
    return null;
  }
};

/**
 * حفظ بيانات التواريخ والأوقات (Step 2 Data)
 * Save dates and times data
 * @param {object} data - بيانات التواريخ والأوقات
 */
export const saveStep2Data = (data) => {
  try {
    cookieStorage.setItem('bookingStep2Data', JSON.stringify(data));
  } catch (error) {
    console.error('Error saving step 2 data:', error);
  }
};

/**
 * استرجاع بيانات التواريخ والأوقات
 * Get dates and times data
 * @returns {object|null} - بيانات التواريخ والأوقات أو null
 */
export const getStep2Data = () => {
  try {
    const data = cookieStorage.getItem('bookingStep2Data');
    return data ? JSON.parse(data) : null;
  } catch (error) {
    console.error('Error getting step 2 data:', error);
    return null;
  }
};

/**
 * حفظ التبويب النشط لكل موقع
 * Save active tab for each location
 * @param {string} locationType - نوع الموقع (pickup أو delivery)
 * @param {string} tab - التبويب النشط (branch أو location)
 */
export const saveLocationTab = (locationType, tab) => {
  try {
    const key = `locationTab_${locationType}`;
    cookieStorage.setItem(key, tab);
  } catch (error) {
    console.error('Error saving location tab:', error);
  }
};

/**
 * استرجاع التبويب النشط للموقع
 * Get active tab for location
 * @param {string} locationType - نوع الموقع (pickup أو delivery)
 * @returns {string|null} - التبويب النشط أو null
 */
export const getLocationTab = (locationType) => {
  try {
    const key = `locationTab_${locationType}`;
    return cookieStorage.getItem(key);
  } catch (error) {
    console.error('Error getting location tab:', error);
    return null;
  }
};

/**
 * حذف جميع بيانات الحجز من cookie storage
 * Clear all booking data from cookie storage
 */
export const clearAllBookingData = () => {
  
  // حذف بيانات الحجز العامة
  // cookieStorage.removeItem('priceDetails');
  // cookieStorage.removeItem('payloadReservation'); 
  // cookieStorage.removeItem('bookingData');
  // cookieStorage.removeItem('carSelection');
  // cookieStorage.removeItem('rentalFilters');
  
  // حذف بيانات الليموزين
  // cookieStorage.removeItem('bookingLim');
  // cookieStorage.removeItem('reservationPriceLimResult');
  // cookieStorage.removeItem('carLimData');
  // cookieStorage.removeItem('receptionData');
  cookieStorage.removeItem('rentalData');
  // cookieStorage.removeItem('otherPlacesFilter');
  
  // حذف بيانات الحجز المؤقتة
  cookieStorage.removeItem('bookingStepData');
  cookieStorage.removeItem('bookingStep2Data');
  cookieStorage.removeItem('locationTab_pickup');
  cookieStorage.removeItem('locationTab_delivery');
};

/**
 * حذف بيانات الحجز العادي (rental) فقط
 * Clear only rental booking data
 */
export const clearRentalBookingData = () => {
  
  
  cookieStorage.removeItem('priceDetails');
  cookieStorage.removeItem('payloadReservation'); 
  cookieStorage.removeItem('bookingData');
  cookieStorage.removeItem('carSelection');
  cookieStorage.removeItem('rentalFilters');
  cookieStorage.removeItem('rentalData');
  
};

/**
 * حذف بيانات الليموزين فقط
 * Clear only limousine booking data
 */
export const clearLimousineBookingData = () => {
  
  cookieStorage.removeItem('bookingLim');
  cookieStorage.removeItem('reservationPriceLimResult');
  cookieStorage.removeItem('carLimData');
  cookieStorage.removeItem('receptionData');
  cookieStorage.removeItem('otherPlacesFilter');
};

/**
 * حذف بيانات المستخدم (authentication data)
 * Clear user authentication data
 */
export const clearUserData = () => {
  
  cookieStorage.removeItem('token');
  cookieStorage.removeItem('user');
  cookieStorage.removeItem('id');
  cookieStorage.removeItem('rememberMobile');
  cookieStorage.removeItem('rememberPassword');
};

/**
 * حذف بيانات محددة من cookie storage
 * Clear specific data from cookie storage
 * @param {string[]} keys - array of keys to remove
 */
export const clearSpecificData = (keys) => {
  
  keys.forEach(key => {
    cookieStorage.removeItem(key);
  });
};

/**
 * حذف جميع البيانات من cookie storage
 * Clear all data from cookie storage
 */
export const clearAllSessionData = () => {
  
  cookieStorage.clear();
};

/**
 * الحصول على قائمة بجميع المفاتيح المحفوظة في cookie storage
 * Get list of all keys stored in cookie storage
 * @returns {string[]} array of keys
 */
export const getAllSessionKeys = () => {
  const keys = [];
  for (let i = 0; i < cookieStorage.length; i++) {
    const key = cookieStorage.key(i);
    if (key) {
      keys.push(key);
    }
  }
  return keys;
};

/**
 * طباعة جميع البيانات المحفوظة في cookie storage (للتطوير فقط)
 * Print all data stored in cookie storage (for development only)
 */
export const debugSessionStorage = () => {
  
  const keys = getAllSessionKeys();
  keys.forEach(key => {
    const value = cookieStorage.getItem(key);
  });
};

/**
 * حذف البيانات بعد إكمال الحجز
 * Clear data after successful booking
 */
export const clearDataAfterBooking = () => {
  
  // حذف بيانات الحجز
  clearAllBookingData();
  
  // حذف بيانات التذكر (remember me)
  cookieStorage.removeItem('rememberMobile');
  cookieStorage.removeItem('rememberPassword');
};

/**
 * حذف البيانات عند تسجيل الخروج
 * Clear data on logout
 */
export const clearDataOnLogout = () => {
  
  // حذف بيانات المستخدم
  clearUserData();
  
  // حذف بيانات الحجز
  clearAllBookingData();
};

/**
 * أمثلة على الاستخدام في المكونات
 * Usage examples in components
 */

/*
// مثال 1: في مكون تسجيل الخروج
// Example 1: In logout component

import { clearDataOnLogout } from '@/utils/sessionUtils';

const LogoutButton = () => {
  const handleLogout = () => {
    clearDataOnLogout();
    router.push('/login');
  };

  return <button onClick={handleLogout}>تسجيل الخروج</button>;
};

// مثال 2: في صفحة إكمال الحجز
// Example 2: In booking completion page

import { clearDataAfterBooking } from '@/utils/sessionUtils';

const BookingComplete = () => {
  const handleBookingSuccess = () => {
    clearDataAfterBooking();
    router.push('/thanks-page');
  };

  return <button onClick={handleBookingSuccess}>إكمال الحجز</button>;
};

// مثال 3: في صفحة إلغاء الحجز
// Example 3: In cancel booking page

import { clearAllBookingData } from '@/utils/sessionUtils';

const CancelBooking = () => {
  const handleCancel = () => {
    clearAllBookingData();
    router.push('/');
  };

  return <button onClick={handleCancel}>إلغاء الحجز</button>;
};

// مثال 4: تنظيف بيانات محددة
// Example 4: Clear specific data

import { clearSpecificData } from '@/utils/sessionUtils';

const ClearPriceData = () => {
  const handleClearPrice = () => {
    clearSpecificData(['priceDetails', 'payloadReservation']);
  };

  return <button onClick={handleClearPrice}>حذف بيانات السعر</button>;
};

// مثال 5: استخدام Hook المتقدم
// Example 5: Using advanced hook

import { useSessionCleanup } from '@/hook/useSessionCleanup';

const AdvancedComponent = () => {
  const { clearDataOnLogoutWithStore } = useSessionCleanup();

  const handleLogout = () => {
    clearDataOnLogoutWithStore(); // يحذف من session storage و Redux store
    router.push('/login');
  };

  return <button onClick={handleLogout}>تسجيل الخروج المتقدم</button>;
};
*/ 