/**
 * Performance Utilities
 * مجموعة من الأدوات لتحسين الأداء في جميع أنحاء التطبيق
 */

/**
 * دالة لتأخير التحميل (Debounce)
 * تستخدم لتقليل عدد استدعاءات الدوال
 */
export function debounce(func, wait = 300) {
  let timeout;
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout);
      func(...args);
    };
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };
}

/**
 * دالة للتقييد (Throttle)
 * تستخدم لتحديد عدد استدعاءات الدالة في فترة زمنية
 */
export function throttle(func, limit = 300) {
  let inThrottle;
  return function executedFunction(...args) {
    if (!inThrottle) {
      func.apply(this, args);
      inThrottle = true;
      setTimeout(() => (inThrottle = false), limit);
    }
  };
}

/**
 * دالة للتحقق من وجود عنصر في viewport
 * تستخدم للتحميل الكسول المخصص
 */
export function isInViewport(element) {
  if (!element) return false;
  const rect = element.getBoundingClientRect();
  return (
    rect.top >= 0 &&
    rect.left >= 0 &&
    rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
    rect.right <= (window.innerWidth || document.documentElement.clientWidth)
  );
}

/**
 * دالة لتحميل الصور بشكل كسول
 * @param {string} src - مصدر الصورة
 * @param {Function} onLoad - دالة يتم استدعاؤها عند تحميل الصورة
 */
export function lazyLoadImage(src, onLoad) {
  const img = new Image();
  img.onload = () => {
    if (onLoad) onLoad(img);
  };
  img.src = src;
  return img;
}

/**
 * دالة لقياس أداء الدالة
 * @param {Function} fn - الدالة المراد قياس أدائها
 * @param {string} label - تسمية للقياس
 */
export function measurePerformance(fn, label = 'Function') {
  const start = performance.now();
  const result = fn();
  const end = performance.now();
  console.log(`${label} took ${(end - start).toFixed(2)}ms`);
  return result;
}

/**
 * دالة للتخزين المؤقت في الذاكرة (Memoization)
 * @param {Function} fn - الدالة المراد حفظ نتائجها
 */
export function memoize(fn) {
  const cache = new Map();
  return function memoized(...args) {
    const key = JSON.stringify(args);
    if (cache.has(key)) {
      return cache.get(key);
    }
    const result = fn.apply(this, args);
    cache.set(key, result);
    return result;
  };
}

/**
 * دالة للتخزين المؤقت مع انتهاء صلاحية (Cache with TTL)
 * @param {number} ttl - مدة الصلاحية بالميلي ثانية
 */
export function createCacheWithTTL(ttl = 5 * 60 * 1000) {
  const cache = new Map();
  
  return {
    get(key) {
      const item = cache.get(key);
      if (!item) return null;
      
      if (Date.now() > item.expiry) {
        cache.delete(key);
        return null;
      }
      
      return item.value;
    },
    
    set(key, value) {
      cache.set(key, {
        value,
        expiry: Date.now() + ttl
      });
    },
    
    clear() {
      cache.clear();
    },
    
    delete(key) {
      cache.delete(key);
    }
  };
}

/**
 * دالة لتحسين الصور
 * @param {string} src - مصدر الصورة
 * @param {object} options - خيارات التحسين
 */
export function optimizeImageUrl(src, options = {}) {
  const {
    width,
    height,
    quality = 80,
    format = 'webp'
  } = options;
  
  // إذا كان الرابط من Next.js Image Optimizer
  if (src.startsWith('/_next/image')) {
    return src;
  }
  
  // إضافة معاملات التحسين للصور الخارجية
  const url = new URL(src, window.location.origin);
  if (width) url.searchParams.set('w', width);
  if (height) url.searchParams.set('h', height);
  if (quality) url.searchParams.set('q', quality);
  if (format) url.searchParams.set('fm', format);
  
  return url.toString();
}

/**
 * دالة للحصول على حجم الشاشة المثالي للصورة
 */
export function getOptimalImageSize() {
  if (typeof window === 'undefined') return { width: 1920, height: 1080 };
  
  const width = window.innerWidth;
  const height = window.innerHeight;
  const dpr = window.devicePixelRatio || 1;
  
  return {
    width: Math.round(width * dpr),
    height: Math.round(height * dpr)
  };
}

/**
 * دالة لتحميل السكريبتات بشكل غير متزامن
 * @param {string} src - مصدر السكريبت
 * @param {object} options - خيارات التحميل
 */
export function loadScriptAsync(src, options = {}) {
  return new Promise((resolve, reject) => {
    const script = document.createElement('script');
    script.src = src;
    script.async = options.async !== false;
    script.defer = options.defer || false;
    
    script.onload = () => resolve(script);
    script.onerror = () => reject(new Error(`Failed to load script: ${src}`));
    
    document.head.appendChild(script);
  });
}

/**
 * دالة للتحقق من سرعة الاتصال
 */
export function getConnectionSpeed() {
  if (typeof navigator === 'undefined' || !navigator.connection) {
    return 'unknown';
  }
  
  const connection = navigator.connection;
  const effectiveType = connection.effectiveType;
  
  // slow-2g, 2g, 3g, 4g
  return effectiveType;
}

/**
 * دالة لتحديد ما إذا كان يجب تحميل موارد عالية الجودة
 */
export function shouldLoadHighQuality() {
  const speed = getConnectionSpeed();
  return speed === '4g' || speed === 'unknown';
}

/**
 * دالة لتحسين استخدام الذاكرة
 * تقوم بتنظيف الكائنات غير المستخدمة
 */
export function cleanupUnusedResources(refs) {
  if (Array.isArray(refs)) {
    refs.forEach(ref => {
      if (ref && ref.current) {
        ref.current = null;
      }
    });
  }
}

/**
 * دالة لقياس Core Web Vitals
 */
export function measureCoreWebVitals() {
  if (typeof window === 'undefined') return;
  
  // قياس FCP (First Contentful Paint)
  const fcpObserver = new PerformanceObserver((entryList) => {
    for (const entry of entryList.getEntries()) {
      console.log('FCP:', entry.startTime);
    }
  });
  
  try {
    fcpObserver.observe({ entryTypes: ['paint'] });
  } catch (e) {
    console.warn('Performance Observer not supported');
  }
  
  // قياس LCP (Largest Contentful Paint)
  const lcpObserver = new PerformanceObserver((entryList) => {
    const entries = entryList.getEntries();
    const lastEntry = entries[entries.length - 1];
    console.log('LCP:', lastEntry.renderTime || lastEntry.loadTime);
  });
  
  try {
    lcpObserver.observe({ entryTypes: ['largest-contentful-paint'] });
  } catch (e) {
    console.warn('LCP Observer not supported');
  }
}

/**
 * Hook مخصص لاستخدام Intersection Observer
 * @param {object} options - خيارات المراقب
 */
export function createIntersectionObserver(callback, options = {}) {
  const {
    threshold = 0,
    root = null,
    rootMargin = '0px'
  } = options;
  
  if (typeof window === 'undefined' || !window.IntersectionObserver) {
    return null;
  }
  
  return new IntersectionObserver(callback, {
    threshold,
    root,
    rootMargin
  });
}

/**
 * دالة لتحسين الأداء عند التمرير
 */
export function optimizeScroll(callback) {
  let ticking = false;
  
  return function optimizedScroll(e) {
    if (!ticking) {
      window.requestAnimationFrame(() => {
        callback(e);
        ticking = false;
      });
      ticking = true;
    }
  };
}

/**
 * دالة لتحميل مكون بشكل كسول مع retry
 * @param {Function} importFn - دالة الاستيراد
 * @param {number} retries - عدد المحاولات
 */
export async function lazyLoadWithRetry(importFn, retries = 3) {
  try {
    return await importFn();
  } catch (error) {
    if (retries === 0) {
      throw error;
    }
    
    // الانتظار قليلاً قبل المحاولة مرة أخرى
    await new Promise(resolve => setTimeout(resolve, 1000));
    return lazyLoadWithRetry(importFn, retries - 1);
  }
}

export default {
  debounce,
  throttle,
  isInViewport,
  lazyLoadImage,
  measurePerformance,
  memoize,
  createCacheWithTTL,
  optimizeImageUrl,
  getOptimalImageSize,
  loadScriptAsync,
  getConnectionSpeed,
  shouldLoadHighQuality,
  cleanupUnusedResources,
  measureCoreWebVitals,
  createIntersectionObserver,
  optimizeScroll,
  lazyLoadWithRetry
};

