/**
 * Utility functions for password validation
 */

/**
 * Check if password meets minimum length requirement
 * @param {string} password - The password to validate
 * @param {number} minLength - Minimum required length (default: 8)
 * @returns {boolean}
 */
export const hasMinLength = (password, minLength = 8) => {
  return password.length >= minLength;
};

/**
 * Check if password contains at least one uppercase letter
 * @param {string} password - The password to validate
 * @returns {boolean}
 */
export const hasUppercase = (password) => {
  return /[A-Z]/.test(password);
};

/**
 * Check if password contains at least one lowercase letter
 * @param {string} password - The password to validate
 * @returns {boolean}
 */
export const hasLowercase = (password) => {
  return /[a-z]/.test(password);
};

/**
 * Check if password contains at least one number
 * @param {string} password - The password to validate
 * @returns {boolean}
 */
export const hasNumber = (password) => {
  return /\d/.test(password);
};

/**
 * Check if password contains at least one special character
 * @param {string} password - The password to validate
 * @returns {boolean}
 */
export const hasSpecialChar = (password) => {
  return /[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password);
};

/**
 * Get all password validation checks
 * @param {string} password - The password to validate
 * @param {number} minLength - Minimum required length (default: 8)
 * @returns {object} Object with boolean values for each validation rule
 */
export const getPasswordChecks = (password, minLength = 8) => {
  return {
    length: hasMinLength(password, minLength),
    uppercase: hasUppercase(password),
    lowercase: hasLowercase(password),
    number: hasNumber(password),
    special: hasSpecialChar(password)
  };
};

/**
 * Check if password meets all validation requirements
 * @param {string} password - The password to validate
 * @param {number} minLength - Minimum required length (default: 8)
 * @returns {boolean}
 */
export const isPasswordValid = (password, minLength = 8) => {
  const checks = getPasswordChecks(password, minLength);
  return Object.values(checks).every(check => check === true);
};
