import { URL } from '@/constants/api';

/**
 * Validates if a location is within service area by checking the nearest branch
 * @param {number} latitude - The latitude coordinate
 * @param {number} longitude - The longitude coordinate
 * @returns {Promise<Object>} - Validation result with status and branch info
 */
export const validateLocation = async (latitude, longitude) => {
  try {
    const endpoint = `/branch/get-nearest-branch?latitude=${latitude}&longitude=${longitude}`;
    const url = URL(endpoint);
    
    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    console.log(data);
    // Check if the API call was successful
    if (data.status && data.message == 'SUCCESS') {
      const { branch, response: responseMessage } = data.data;

      console.log(branch, responseMessage);
      // Determine if location is valid based on response
      if (branch && responseMessage == "Delivery Service Allowed For This Location") {
        return {
          isValid: true,
          branch: branch,
          message: responseMessage,
          error: null
        };
      } else if (responseMessage == "Delivery Service Not Allowed For This Location") {
        return {
          isValid: false,
          branch: null,
          message: responseMessage,
          error: 'DELIVERY_NOT_ALLOWED'
        };
      } else if (responseMessage == "No Branch Found For This Location") {
        return {
          isValid: false,
          branch: null,
          message: responseMessage,
          error: 'NO_BRANCH_FOUND'
        };
      } else {
        return {
          isValid: false,
          branch: null,
          message: responseMessage || 'Unknown error',
          error: 'UNKNOWN_ERROR'
        };
      }
    } else {
      return {
        isValid: false,
        branch: null,
        message: data.message || 'API call failed',
        error: 'API_ERROR'
      };
    }
  } catch (error) {
    console.error('Location validation error:', error);
    return {
      isValid: false,
      branch: null,
      message: 'Failed to validate location',
      error: 'NETWORK_ERROR'
    };
  }
};

/**
 * Gets user-friendly error message based on validation result
 * @param {Object} validationResult - Result from validateLocation function
 * @param {string} locale - Current locale (ar/en)
 * @returns {string} - User-friendly error message
 */
export const getLocationErrorMessage = (validationResult, locale = 'ar') => {
  const { error, message } = validationResult;
  
  const errorMessages = {
    ar: {
      DELIVERY_NOT_ALLOWED: 'خدمة التوصيل غير متاحة في هذا الموقع',
      NO_BRANCH_FOUND: 'لا يوجد فرع قريب من هذا الموقع',
      UNKNOWN_ERROR: 'خطأ غير معروف في التحقق من الموقع',
      API_ERROR: 'فشل في التحقق من صحة الموقع',
      NETWORK_ERROR: 'خطأ في الاتصال، يرجى المحاولة مرة أخرى'
    },
    en: {
      DELIVERY_NOT_ALLOWED: 'Delivery service is not available at this location',
      NO_BRANCH_FOUND: 'No branch found near this location',
      UNKNOWN_ERROR: 'Unknown error occurred while validating location',
      API_ERROR: 'Failed to validate location',
      NETWORK_ERROR: 'Connection error, please try again'
    }
  };

  return errorMessages[locale]?.[error] || message || errorMessages[locale]?.UNKNOWN_ERROR;
};
