/**
 * Limousine Location Validation
 * للتحقق من المواقع الخاصة بخدمة الليموزين بدون استخدام API
 * يتم التحقق فقط أن الموقع داخل حدود المدينة المحددة (جدة أو مكة)
 */

// حدود المدن السعودية
const CITY_BOUNDS = {
  JEDDAH: {
    name: 'جدة',
    nameEn: 'Jeddah',
    center: { lat: 21.5433, lng: 39.1728 },
    bounds: {
      north: 21.8,
      south: 21.3,
      east: 39.4,
      west: 39.0
    },
    // نصف قطر تقريبي بالكيلومترات
    radiusKm: 40
  },
  MAKKAH: {
    name: 'مكة المكرمة',
    nameEn: 'Makkah',
    center: { lat: 21.4225, lng: 39.8262 },
    bounds: {
      north: 21.6,
      south: 21.2,
      east: 40.1,
      west: 39.6
    },
    radiusKm: 35
  }
};

/**
 * حساب المسافة بين نقطتين باستخدام Haversine formula
 * @param {number} lat1 - خط العرض للنقطة الأولى
 * @param {number} lon1 - خط الطول للنقطة الأولى
 * @param {number} lat2 - خط العرض للنقطة الثانية
 * @param {number} lon2 - خط الطول للنقطة الثانية
 * @returns {number} - المسافة بالكيلومترات
 */
const calculateDistance = (lat1, lon1, lat2, lon2) => {
  const R = 6371; // نصف قطر الأرض بالكيلومترات
  const dLat = toRad(lat2 - lat1);
  const dLon = toRad(lon2 - lon1);
  const a =
    Math.sin(dLat / 2) * Math.sin(dLat / 2) +
    Math.cos(toRad(lat1)) * Math.cos(toRad(lat2)) * Math.sin(dLon / 2) * Math.sin(dLon / 2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
  const distance = R * c;
  return distance;
};

const toRad = (value) => {
  return (value * Math.PI) / 180;
};

/**
 * التحقق من أن الموقع داخل حدود المدينة المحددة
 * @param {number} latitude - خط العرض
 * @param {number} longitude - خط الطول
 * @param {string} cityId - معرف المدينة (JEDDAH أو MAKKAH)
 * @returns {Object} - نتيجة التحقق
 */
export const validateLimousineLocation = (latitude, longitude, cityId) => {
  // التحقق من وجود المدينة
  if (!cityId || !CITY_BOUNDS[cityId]) {
    return {
      isValid: false,
      city: null,
      message: 'يرجى اختيار المدينة أولاً',
      error: 'NO_CITY_SELECTED'
    };
  }

  const city = CITY_BOUNDS[cityId];
  
  // التحقق من الإحداثيات
  if (!latitude || !longitude || isNaN(latitude) || isNaN(longitude)) {
    return {
      isValid: false,
      city: city,
      message: 'إحداثيات غير صالحة',
      error: 'INVALID_COORDINATES'
    };
  }

  // التحقق باستخدام Bounding Box
  const isWithinBounds =
    latitude >= city.bounds.south &&
    latitude <= city.bounds.north &&
    longitude >= city.bounds.west &&
    longitude <= city.bounds.east;

  // التحقق باستخدام المسافة من مركز المدينة
  const distance = calculateDistance(
    city.center.lat,
    city.center.lng,
    latitude,
    longitude
  );
  
  const isWithinRadius = distance <= city.radiusKm;

  // الموقع صالح إذا كان ضمن الحدود أو ضمن نصف القطر
  const isValid = isWithinBounds || isWithinRadius;

  if (isValid) {
    return {
      isValid: true,
      city: city,
      distance: distance.toFixed(2),
      message: `الموقع داخل ${city.name}`,
      error: null
    };
  } else {
    return {
      isValid: false,
      city: city,
      distance: distance.toFixed(2),
      message: `الموقع خارج حدود ${city.name}. المسافة من المركز: ${distance.toFixed(2)} كم`,
      error: 'OUTSIDE_CITY_BOUNDS'
    };
  }
};

/**
 * الحصول على رسالة الخطأ بناءً على نتيجة التحقق
 * @param {Object} validationResult - نتيجة التحقق
 * @param {string} locale - اللغة (ar/en)
 * @returns {string} - رسالة الخطأ
 */
export const getLimousineLocationErrorMessage = (validationResult, locale = 'ar') => {
  const { error, city, distance } = validationResult;

  const errorMessages = {
    ar: {
      NO_CITY_SELECTED: 'يرجى اختيار المدينة أولاً',
      INVALID_COORDINATES: 'إحداثيات غير صالحة، يرجى اختيار موقع على الخريطة',
      OUTSIDE_CITY_BOUNDS: city
        ? `الموقع المحدد خارج حدود ${city.name}. يرجى اختيار موقع داخل ${city.name}`
        : 'الموقع المحدد خارج حدود المدينة المختارة',
      UNKNOWN_ERROR: 'خطأ في التحقق من الموقع'
    },
    en: {
      NO_CITY_SELECTED: 'Please select a city first',
      INVALID_COORDINATES: 'Invalid coordinates, please select a location on the map',
      OUTSIDE_CITY_BOUNDS: city
        ? `Selected location is outside ${city.nameEn} boundaries. Please select a location within ${city.nameEn}`
        : 'Selected location is outside the chosen city boundaries',
      UNKNOWN_ERROR: 'Error validating location'
    }
  };

  return errorMessages[locale]?.[error] || validationResult.message || errorMessages[locale]?.UNKNOWN_ERROR;
};

/**
 * الحصول على مركز الخريطة بناءً على المدينة المحددة
 * @param {string} cityId - معرف المدينة (JEDDAH أو MAKKAH)
 * @returns {Object} - إحداثيات مركز المدينة
 */
export const getCityCenter = (cityId) => {
  if (!cityId || !CITY_BOUNDS[cityId]) {
    // Default to Jeddah if no city selected
    return CITY_BOUNDS.JEDDAH.center;
  }
  return CITY_BOUNDS[cityId].center;
};

/**
 * الحصول على معلومات المدينة
 * @param {string} cityId - معرف المدينة
 * @returns {Object} - معلومات المدينة
 */
export const getCityInfo = (cityId) => {
  return CITY_BOUNDS[cityId] || null;
};
