/**
 * Cookie Storage Utility
 * Provides a sessionStorage-like API using cookies
 * مساعد تخزين الكوكيز - يوفر واجهة مشابهة لـ sessionStorage باستخدام الكوكيز
 */
'use client';

import Cookies from 'js-cookie';

// Default cookie options
const DEFAULT_OPTIONS = {
  expires: 1, // 1 day
  path: '/',
  sameSite: 'Lax',
  secure: process.env.NODE_ENV === 'production',
};

/**
 * Cookie Storage class that mimics sessionStorage API
 */
class CookieStorage {
  /**
   * Set item in cookie storage
   * @param {string} key - The key
   * @param {string} value - The value to store
   * @param {object} options - Optional cookie options
   */
  setItem(key, value, options = {}) {
    try {
      Cookies.set(key, value, { ...DEFAULT_OPTIONS, ...options });
    } catch (error) {
      // Error setting cookie
    }
  }

  /**
   * Get item from cookie storage
   * @param {string} key - The key
   * @returns {string|null} The value or null if not found
   */
  getItem(key) {
    try {
      return Cookies.get(key) || null;
    } catch (error) {
      // Error getting cookie
      return null;
    }
  }

  /**
   * Remove item from cookie storage
   * @param {string} key - The key to remove
   */
  removeItem(key) {
    try {
      Cookies.remove(key, { path: '/' });
    } catch (error) {
      // Error removing cookie
    }
  }

  /**
   * Clear all cookies
   * Note: This only clears cookies that were set by this domain
   */
  clear() {
    try {
      const allCookies = Cookies.get();
      Object.keys(allCookies).forEach(key => {
        Cookies.remove(key, { path: '/' });
      });
    } catch (error) {
      // Error clearing cookies
    }
  }

  /**
   * Get the key at a specific index
   * @param {number} index - The index
   * @returns {string|null} The key or null
   */
  key(index) {
    try {
      const keys = Object.keys(Cookies.get());
      return keys[index] || null;
    } catch (error) {
      // Error getting key
      return null;
    }
  }

  /**
   * Get the number of items in storage
   * @returns {number} The number of cookies
   */
  get length() {
    try {
      return Object.keys(Cookies.get()).length;
    } catch (error) {
      // Error getting length
      return 0;
    }
  }
}

// Create singleton instance
const cookieStorage = new CookieStorage();

export default cookieStorage;

// Export individual functions for convenience
export const setItem = (key, value, options) => cookieStorage.setItem(key, value, options);
export const getItem = (key) => cookieStorage.getItem(key);
export const removeItem = (key) => cookieStorage.removeItem(key);
export const clear = () => cookieStorage.clear();
export const key = (index) => cookieStorage.key(index);

