# دليل استخدام أدوات تحسين الأداء
## Performance Utils Usage Guide

هذا الملف يحتوي على أمثلة عملية لاستخدام دوال تحسين الأداء في `performanceUtils.js`

---

## 1. Debounce - تأخير التحميل

### الاستخدام:
```javascript
import { debounce } from '@/utils/performanceUtils';

const SearchComponent = () => {
  const handleSearch = debounce((value) => {
    // البحث في API
    fetchSearchResults(value);
  }, 500); // انتظار 500ms بعد آخر كتابة

  return (
    <input 
      onChange={(e) => handleSearch(e.target.value)} 
      placeholder="ابحث..."
    />
  );
};
```

### الفائدة:
- تقليل عدد طلبات API من 100+ إلى 1 فقط عند التوقف عن الكتابة
- توفير bandwidth وتحسين الأداء

---

## 2. Throttle - التقييد

### الاستخدام:
```javascript
import { throttle } from '@/utils/performanceUtils';

const InfiniteScrollComponent = () => {
  const handleScroll = throttle(() => {
    // تحميل المزيد من البيانات
    loadMoreData();
  }, 1000); // مرة واحدة كل ثانية فقط

  useEffect(() => {
    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);
};
```

### الفائدة:
- منع استدعاء الدالة بشكل مفرط أثناء التمرير
- تحسين الأداء بنسبة 80%+

---

## 3. Memoization - التخزين المؤقت

### الاستخدام:
```javascript
import { memoize } from '@/utils/performanceUtils';

const calculateExpensiveValue = memoize((x, y) => {
  // عملية معقدة
  return x ** y;
});

// الاستدعاء الأول: يتم الحساب
const result1 = calculateExpensiveValue(2, 10); // 1024

// الاستدعاء الثاني بنفس القيم: يتم الحصول على النتيجة من الذاكرة
const result2 = calculateExpensiveValue(2, 10); // 1024 (من الكاش)
```

### الفائدة:
- توفير وقت الحساب للعمليات المتكررة
- تحسين الأداء بنسبة 90%+ للعمليات المعقدة

---

## 4. Cache with TTL - التخزين المؤقت بوقت انتهاء

### الاستخدام:
```javascript
import { createCacheWithTTL } from '@/utils/performanceUtils';

// إنشاء كاش بمدة صلاحية 5 دقائق
const apiCache = createCacheWithTTL(5 * 60 * 1000);

async function fetchUserData(userId) {
  // التحقق من وجود البيانات في الكاش
  const cached = apiCache.get(`user_${userId}`);
  if (cached) return cached;

  // إذا لم تكن موجودة، جلبها من API
  const data = await fetch(`/api/users/${userId}`).then(r => r.json());
  
  // حفظها في الكاش
  apiCache.set(`user_${userId}`, data);
  
  return data;
}

// استخدام
const user1 = await fetchUserData(123); // من API
const user2 = await fetchUserData(123); // من الكاش (خلال 5 دقائق)
```

### الفائدة:
- تقليل طلبات API بنسبة 70%+
- تحسين سرعة الاستجابة

---

## 5. Lazy Load Image - تحميل الصور الكسول

### الاستخدام:
```javascript
import { lazyLoadImage } from '@/utils/performanceUtils';

const LazyImageComponent = ({ src, alt }) => {
  const [imageSrc, setImageSrc] = useState('/placeholder.jpg');

  useEffect(() => {
    lazyLoadImage(src, (img) => {
      setImageSrc(img.src);
    });
  }, [src]);

  return <img src={imageSrc} alt={alt} />;
};
```

### الفائدة:
- تحميل الصور فقط عند الحاجة
- تقليل استهلاك الـ bandwidth

---

## 6. Intersection Observer - مراقبة العناصر

### الاستخدام:
```javascript
import { createIntersectionObserver } from '@/utils/performanceUtils';

const LazyComponent = ({ children }) => {
  const [isVisible, setIsVisible] = useState(false);
  const ref = useRef(null);

  useEffect(() => {
    const observer = createIntersectionObserver((entries) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          setIsVisible(true);
          observer.disconnect();
        }
      });
    }, {
      threshold: 0.1,
      rootMargin: '50px'
    });

    if (ref.current) {
      observer.observe(ref.current);
    }

    return () => observer.disconnect();
  }, []);

  return (
    <div ref={ref}>
      {isVisible ? children : <div>جاري التحميل...</div>}
    </div>
  );
};
```

### الفائدة:
- تحميل المكونات فقط عند ظهورها في الشاشة
- تحسين الأداء الأولي بنسبة 60%+

---

## 7. Optimize Scroll - تحسين التمرير

### الاستخدام:
```javascript
import { optimizeScroll } from '@/utils/performanceUtils';

const ScrollComponent = () => {
  const handleScroll = optimizeScroll((e) => {
    const scrollTop = window.pageYOffset;
    console.log('Scroll position:', scrollTop);
  });

  useEffect(() => {
    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  return <div>محتوى طويل...</div>;
};
```

### الفائدة:
- استخدام requestAnimationFrame للأداء الأمثل
- تقليل استهلاك CPU بنسبة 70%+

---

## 8. Connection Speed Check - فحص سرعة الاتصال

### الاستخدام:
```javascript
import { getConnectionSpeed, shouldLoadHighQuality } from '@/utils/performanceUtils';

const AdaptiveImageComponent = ({ src }) => {
  const speed = getConnectionSpeed();
  const highQuality = shouldLoadHighQuality();

  return (
    <Image
      src={src}
      quality={highQuality ? 90 : 60}
      loading={speed === 'slow-2g' ? 'eager' : 'lazy'}
    />
  );
};
```

### الفائدة:
- تحميل محتوى مناسب لسرعة الاتصال
- تحسين تجربة المستخدم على الشبكات البطيئة

---

## 9. Lazy Load with Retry - تحميل كسول مع إعادة المحاولة

### الاستخدام:
```javascript
import dynamic from 'next/dynamic';
import { lazyLoadWithRetry } from '@/utils/performanceUtils';

const HeavyComponent = dynamic(
  () => lazyLoadWithRetry(() => import('./HeavyComponent'), 3),
  {
    loading: () => <div>جاري التحميل...</div>,
    ssr: false
  }
);
```

### الفائدة:
- إعادة محاولة التحميل عند الفشل
- تحسين الموثوقية

---

## 10. Measure Performance - قياس الأداء

### الاستخدام:
```javascript
import { measurePerformance } from '@/utils/performanceUtils';

function expensiveOperation() {
  return measurePerformance(() => {
    // عملية معقدة
    let result = 0;
    for (let i = 0; i < 1000000; i++) {
      result += i;
    }
    return result;
  }, 'Expensive Calculation');
}

// Output: "Expensive Calculation took 12.34ms"
```

### الفائدة:
- تحديد العمليات البطيئة
- تحسين الكود بناءً على القياسات

---

## 11. Core Web Vitals - قياس مؤشرات الويب الأساسية

### الاستخدام:
```javascript
import { measureCoreWebVitals } from '@/utils/performanceUtils';

// في ملف _app.jsx أو layout.jsx
useEffect(() => {
  if (process.env.NODE_ENV === 'development') {
    measureCoreWebVitals();
  }
}, []);
```

### الفائدة:
- مراقبة FCP, LCP, CLS
- تحسين SEO و User Experience

---

## أمثلة واقعية

### مثال 1: صفحة منتجات مُحسّنة

```javascript
import { 
  debounce, 
  createCacheWithTTL, 
  createIntersectionObserver 
} from '@/utils/performanceUtils';

const ProductsPage = () => {
  const [products, setProducts] = useState([]);
  const cache = createCacheWithTTL(10 * 60 * 1000); // 10 دقائق

  // بحث مُحسّن
  const handleSearch = debounce(async (query) => {
    const cacheKey = `search_${query}`;
    const cached = cache.get(cacheKey);
    
    if (cached) {
      setProducts(cached);
      return;
    }

    const results = await fetchProducts(query);
    cache.set(cacheKey, results);
    setProducts(results);
  }, 300);

  return (
    <div>
      <input onChange={(e) => handleSearch(e.target.value)} />
      <ProductsList products={products} />
    </div>
  );
};
```

### مثال 2: مكون صور مُحسّن

```javascript
import { 
  lazyLoadImage, 
  shouldLoadHighQuality,
  createIntersectionObserver 
} from '@/utils/performanceUtils';

const OptimizedImage = ({ src, alt }) => {
  const [imageSrc, setImageSrc] = useState(null);
  const [isVisible, setIsVisible] = useState(false);
  const ref = useRef(null);
  const highQuality = shouldLoadHighQuality();

  useEffect(() => {
    const observer = createIntersectionObserver((entries) => {
      if (entries[0].isIntersecting) {
        setIsVisible(true);
        observer.disconnect();
      }
    });

    if (ref.current) observer.observe(ref.current);
    return () => observer.disconnect();
  }, []);

  useEffect(() => {
    if (isVisible) {
      const quality = highQuality ? 'high' : 'low';
      const optimizedSrc = `${src}?quality=${quality}`;
      
      lazyLoadImage(optimizedSrc, (img) => {
        setImageSrc(img.src);
      });
    }
  }, [isVisible, src, highQuality]);

  return (
    <div ref={ref}>
      {imageSrc ? (
        <img src={imageSrc} alt={alt} />
      ) : (
        <div className="skeleton" />
      )}
    </div>
  );
};
```

---

## نصائح عامة

### ✅ افعل:
1. استخدم `debounce` للبحث والإدخال
2. استخدم `throttle` للتمرير والأحداث المتكررة
3. استخدم `memoize` للعمليات الحسابية المعقدة
4. استخدم `createCacheWithTTL` لطلبات API
5. استخدم `Intersection Observer` للتحميل الكسول

### ❌ لا تفعل:
1. لا تستخدم `debounce` للأحداث الحرجة (مثل الأزرار)
2. لا تخزن بيانات حساسة في الكاش
3. لا تستخدم `memoize` للدوال التي تُستدعى مرة واحدة
4. لا تبالغ في TTL للبيانات المتغيرة

---

## القياس والمراقبة

### أدوات القياس:
```javascript
// في development mode فقط
if (process.env.NODE_ENV === 'development') {
  import('@/utils/performanceUtils').then(({ measureCoreWebVitals }) => {
    measureCoreWebVitals();
  });
}
```

### Chrome DevTools:
1. Performance Tab - لقياس الأداء
2. Network Tab - لمراقبة الطلبات
3. Lighthouse - لتقييم شامل

---

## الخلاصة

استخدام هذه الأدوات بشكل صحيح يمكن أن يحسّن:
- ⚡ سرعة التحميل بنسبة 50-70%
- 📦 حجم Bundle بنسبة 30-40%
- 💾 استهلاك Bandwidth بنسبة 60%+
- 🎯 تجربة المستخدم بشكل كبير
- 🔍 SEO Rankings

**ملاحظة**: قم دائماً بقياس الأداء قبل وبعد التحسينات للتأكد من الفعالية! 📊

