import { createSlice } from '@reduxjs/toolkit';

// دالة لقراءة البيانات من sessionStorage
const loadFromSessionStorage = (key, defaultValue) => {
  if (typeof window === 'undefined') return defaultValue;
  try {
    const item = sessionStorage.getItem(key);
    return item ? JSON.parse(item) : defaultValue;
  } catch (error) {
    return defaultValue;
  }
};

// دالة لحفظ البيانات في sessionStorage
const saveToSessionStorage = (key, value) => {
  if (typeof window === 'undefined') return;
  try {
    if (value === null || value === undefined) {
      sessionStorage.removeItem(key);
    } else {
      sessionStorage.setItem(key, JSON.stringify(value));
    }
  } catch (error) {
    // Handle error silently
  }
};

const initialState = {
  duration: 'daily',
  deliveryType: 'branch',
  startTime: loadFromSessionStorage('rental_startTime', null),
  selectedRange: loadFromSessionStorage('rental_selectedRange', []),
  selectedCarTypes: [],
  selectedCityId: loadFromSessionStorage('rental_selectedCityId', null),
  selectedBranchId: loadFromSessionStorage('rental_selectedBranchId', null),
  rentalCarData: {},
  savedStartDate: loadFromSessionStorage('rental_savedStartDate', null),
  savedEndDate: loadFromSessionStorage('rental_savedEndDate', null),
  savedStartTime: loadFromSessionStorage('rental_savedStartTime', null),
  // بيانات موقع الاستلام والتوصيل
  pickupAddress: null,
  pickupLatitude: null,
  pickupLongitude: null,
  deliveryAddress: null,
  deliveryLatitude: null,
  deliveryLongitude: null,
};

const rentalSlice = createSlice({
  name: 'rental',
  initialState,
  reducers: {
    setDuration: (state, action) => {
      state.duration = action.payload;
    },
    setDeliveryType: (state, action) => {
      state.deliveryType = action.payload;
    },
    setSelectedRange: (state, action) => {
      // Ensure we only store serializable values (strings, numbers, arrays, objects)
      state.selectedRange = action.payload || [];
      saveToSessionStorage('rental_selectedRange', state.selectedRange);
    },
    setStartTime: (state, action) => {
      state.startTime = action.payload;
      saveToSessionStorage('rental_startTime', action.payload);
    },
    setSelectedCarTypes: (state, action) => {
      state.selectedCarTypes = action.payload;
    },
    setSelectedCityId: (state, action) => {
      state.selectedCityId = action.payload;
      saveToSessionStorage('rental_selectedCityId', action.payload);
    },
    setSelectedBranchId: (state, action) => {
      state.selectedBranchId = action.payload;
      saveToSessionStorage('rental_selectedBranchId', action.payload);
    },
    setRentalCarData: (state, action) => {
      state.rentalCarData = action.payload;
    },
    setSavedStartDate: (state, action) => {
      state.savedStartDate = action.payload;
      saveToSessionStorage('rental_savedStartDate', action.payload);
    },
    setSavedEndDate: (state, action) => {
      state.savedEndDate = action.payload;
      saveToSessionStorage('rental_savedEndDate', action.payload);
    },
    setSavedStartTime: (state, action) => {
      state.savedStartTime = action.payload;
      saveToSessionStorage('rental_savedStartTime', action.payload);
    },
    setPickupLocation: (state, action) => {
      const { address, latitude, longitude } = action.payload;
      state.pickupAddress = address;
      state.pickupLatitude = latitude;
      state.pickupLongitude = longitude;
    },
    setDeliveryLocation: (state, action) => {
      const { address, latitude, longitude } = action.payload;
      state.deliveryAddress = address;
      state.deliveryLatitude = latitude;
      state.deliveryLongitude = longitude;
    },
    clearRentalData: (state) => {
      state.duration = 'daily';
      state.deliveryType = 'branch';
      state.startTime = null;
      state.selectedRange = [];
      state.selectedCarTypes = [];
      state.selectedCityId = null;
      state.selectedBranchId = null;
      state.savedStartDate = null;
      state.savedEndDate = null;
      state.savedStartTime = null;
      state.pickupAddress = null;
      state.pickupLatitude = null;
      state.pickupLongitude = null;
      state.deliveryAddress = null;
      state.deliveryLatitude = null;
      state.deliveryLongitude = null;
      
      // مسح البيانات من sessionStorage أيضاً
      if (typeof window !== 'undefined') {
        sessionStorage.removeItem('rental_startTime');
        sessionStorage.removeItem('rental_selectedRange');
        sessionStorage.removeItem('rental_selectedCityId');
        sessionStorage.removeItem('rental_selectedBranchId');
        sessionStorage.removeItem('rental_savedStartDate');
        sessionStorage.removeItem('rental_savedEndDate');
        sessionStorage.removeItem('rental_savedStartTime');
      }
    },
  },
});

export const { setStartTime, setDuration, setDeliveryType, setSelectedRange, setSelectedCarTypes, setSelectedCityId, setSelectedBranchId, clearRentalData, setRentalCarData, setSavedStartDate, setSavedEndDate, setSavedStartTime, setPickupLocation, setDeliveryLocation } = rentalSlice.actions;
export default rentalSlice.reducer;
