import cookieStorage from '@/utils/cookieStorage';
import { createAsyncThunk, createSlice } from '@reduxjs/toolkit';
import { URL } from '../../constants/api';

export const getProfile = createAsyncThunk('get/profile', async (id) => {
    const token = cookieStorage.getItem('token');
    const url = URL('/client/get-data');

    const response = await fetch(url, {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json',
            Authorization: `Bearer ${token}`
        },
    });

    if (!response.ok) {
        throw new Error('Failed to fetch profile');
    }
    const { data } = await response.json();
    localStorage.setItem('membershipId', data.membershipId);
    return data;
});

// Upload image to viganium.co/uploads
export const uploadImage = createAsyncThunk('upload/image', async (file) => {
    const formData = new FormData();
    formData.append('file', file); 
    const response = await fetch('https://viganium.co/uploads', {
        method: 'POST',
        body: formData,
    });

    if (!response.ok) {
        throw new Error('Failed to upload image');
    }

    const filename = await response.text();
    return filename.trim(); // Remove any whitespace
});

// Update profile with uploaded image filename
export const updateProfile = createAsyncThunk('update/profile', async (profileData) => {
    const token = cookieStorage.getItem('token');
    const url = URL('/client/update');

    
    const response = await fetch(url, {
        method: 'PATCH',
        headers: {
            'Content-Type': 'application/json',
            Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(profileData),
    });

    if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || 'Failed to update profile');
    }

    return await response.json();
});

const initialState = {
    user: [],
    loading: false,
    error: null,
    uploadLoading: false,
    updateLoading: false,
    uploadError: null,
    updateError: null,
};

const profileSlice = createSlice({
    name: 'profile',
    initialState,
    extraReducers: (builder) => {
        builder
            // Get Profile
            .addCase(getProfile.fulfilled, (state, action) => {
                state.user = action.payload;
                state.loading = false;
            })
            .addCase(getProfile.pending, (state) => {
                state.loading = true;
            })
            .addCase(getProfile.rejected, (state, action) => {
                state.loading = false;
                state.error = action.error.message;
            })
            // Upload Image
            .addCase(uploadImage.fulfilled, (state, action) => {
                state.uploadLoading = false;
                state.uploadError = null;
            })
            .addCase(uploadImage.pending, (state) => {
                state.uploadLoading = true;
                state.uploadError = null;
            })
            .addCase(uploadImage.rejected, (state, action) => {
                state.uploadLoading = false;
                state.uploadError = action.error.message;
            })
            // Update Profile
            .addCase(updateProfile.fulfilled, (state, action) => {
                state.updateLoading = false;
                state.updateError = null;
                // Update user data with new profile info
                state.user = { ...state.user, ...action.payload.data };
            })
            .addCase(updateProfile.pending, (state) => {
                state.updateLoading = true;
                state.updateError = null;
            })
            .addCase(updateProfile.rejected, (state, action) => {
                state.updateLoading = false;
                state.updateError = action.error.message;
            });
    },
});

export const profileReducer = profileSlice.reducer;
