// src/lib/auth/forgotPasswordSlice.js
import { AUTH_URL } from '@/constants/api';
import { createAsyncThunk, createSlice } from '@reduxjs/toolkit';

// Send OTP for forgot password
export const sendOTP = createAsyncThunk(
  'forgotPassword/sendOTP',
  async ({ channel, mobile, email }, { rejectWithValue }) => {
    try {
      const payload = { channel };

      if (channel === 'WHATSAPP') {
        payload.mobile = mobile.replace(/\s/g, '');
      } else {
        payload.email = email;
      }

      const response = await fetch(AUTH_URL('/forget-password'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload),
      });

      const data = await response.json();

      if (!response.ok || data.status !== true) {
        return rejectWithValue(data.message || "Failed to send OTP");
      }

      return data.data; // This should return clientId
    } catch (error) {
      return rejectWithValue("Network error");
    }
  }
);
// Send OTP for forgot password
export const sendOTPAccountRecovery = createAsyncThunk(
  'forgotPassword/sendOTPAccountRecovery',
  async ({ channel, mobile, email }, { rejectWithValue }) => {
    try {
      const payload = { channel };

      if (channel === 'WHATSAPP') {
        payload.mobile = mobile.replace(/\s/g, '');
      } else {
        payload.email = email;
      }

      const response = await fetch(AUTH_URL('/account-recovery'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload),
      });

      const data = await response.json();

      if (!response.ok || data.status !== true) {
        return rejectWithValue(data.message || "Failed to send OTP");
      }

      return data.data; // This should return clientId
    } catch (error) {
      return rejectWithValue("Network error");
    }
  }
);

// Verify OTP
export const verifyOTP = createAsyncThunk(
  'forgotPassword/verifyOTP',
  async ({ clientId, code }, { rejectWithValue }) => {
    try {
      const response = await fetch(AUTH_URL('/reset-password-verification'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ clientId, code }),
      });

      const data = await response.json();

      if (!response.ok || data.status !== true) {
        return rejectWithValue(data.message || "OTP verification failed");
      }

      return data.data;
    } catch (error) {
      return rejectWithValue("Network error");
    }
  }
);

// Verify OTP for verify account recovery
export const verifyOTPAccountRecovery = createAsyncThunk(
  'forgotPassword/verifyOTPAccountRecovery',
  async ({ email, mobile, code }, { rejectWithValue }) => {
    try {
      const payload = { code };
      
      if (email) {
        payload.email = email;
        payload.mobile = null;
      } else if (mobile) {
        payload.mobile = mobile.replace(/\s+/g, '');
        payload.email = null;
      }

      const response = await fetch(AUTH_URL('/verify-account-recovery'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload),
      });

      const data = await response.json();

      if (!response.ok || data.status !== true) {
        return rejectWithValue(data.message || "OTP verification failed");
      }

      return data.data;
    } catch (error) {
      return rejectWithValue("Network error");
    }
  }
);

// Reset Password
export const resetPassword = createAsyncThunk(
  'forgotPassword/resetPassword',
  async ({ clientId, password }, { rejectWithValue }) => {
    try {
      const response = await fetch(AUTH_URL('/reset-password'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ clientId, password }),
      });

      const data = await response.json();

      if (!response.ok || data.status !== true) {
        return rejectWithValue(data.message || "Password reset failed");
      }

      return data.data;
    } catch (error) {
      return rejectWithValue("Network error");
    }
  }
);

// Resend OTP
export const resendOTP = createAsyncThunk(
  'forgotPassword/resendOTP',
  async ({ clientId, channel }, { rejectWithValue }) => {
    try {
      const response = await fetch(AUTH_URL('/resend'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ clientId, channel }),
      });

      const data = await response.json();

      if (!response.ok || data.status !== true) {
        return rejectWithValue(data.message || "Failed to resend OTP");
      }

      return data.data;
    } catch (error) {
      return rejectWithValue("Network error");
    }
  }
);

// Resend OTP for account recovery
export const resendOTPAccountRecovery = createAsyncThunk(
  'forgotPassword/resendOTPAccountRecovery',
  async ({ clientId, channel }, { rejectWithValue }) => {
    try {
      const response = await fetch(AUTH_URL('/resend-account-recovery'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ clientId, channel }),
      });

      const data = await response.json();

      if (!response.ok || data.status !== true) {
        return rejectWithValue(data.message || "Failed to resend OTP");
      }

      return data.data;
    } catch (error) {
      return rejectWithValue("Network error");
    }
  }
);

const forgotPasswordSlice = createSlice({
  name: 'forgotPassword',
  initialState: {
    clientId: null,
    loading: false,
    error: null,
    success: false,
  },
  reducers: {
    clearError: (state) => {
      state.error = null;
    },
    clearSuccess: (state) => {
      state.success = false;
    },
    resetState: (state) => {
      state.loading = false;
      state.error = null;
      state.success = false;
    },
  },
  extraReducers: (builder) => {
    builder
      // Send OTP
      .addCase(sendOTP.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(sendOTP.fulfilled, (state, action) => {
        state.loading = false;
        state.clientId = action.payload;
        state.success = true;
      })
      .addCase(sendOTP.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload;
      })
      // Verify OTP
      .addCase(verifyOTP.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(verifyOTP.fulfilled, (state) => {
        state.loading = false;
        state.success = true;
      })
      .addCase(verifyOTP.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload;
      })
      // Reset Password
      .addCase(resetPassword.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(resetPassword.fulfilled, (state) => {
        state.loading = false;
        state.success = true;
      })
      .addCase(resetPassword.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload;
      })
      // Resend OTP
      .addCase(resendOTP.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(resendOTP.fulfilled, (state) => {
        state.loading = false;
        state.success = true;
      })
      .addCase(resendOTP.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload;
      })
      // Send OTP Account Recovery
      .addCase(sendOTPAccountRecovery.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(sendOTPAccountRecovery.fulfilled, (state, action) => {
        state.loading = false;
        state.clientId = action.payload;
        state.success = true;
      })
      .addCase(sendOTPAccountRecovery.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload;
      })
      // Verify OTP Account Recovery
      .addCase(verifyOTPAccountRecovery.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(verifyOTPAccountRecovery.fulfilled, (state) => {
        state.loading = false;
        state.success = true;
      })
      .addCase(verifyOTPAccountRecovery.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload;
      })
      // Resend OTP Account Recovery
      .addCase(resendOTPAccountRecovery.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(resendOTPAccountRecovery.fulfilled, (state) => {
        state.loading = false;
        state.success = true;
      })
      .addCase(resendOTPAccountRecovery.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload;
      });
  },
});

export const { clearError, clearSuccess, resetState } = forgotPasswordSlice.actions;
export default forgotPasswordSlice.reducer;
