'use client';

import { useEffect, useRef } from 'react';

/**
 * Hook مخصص لحفظ واسترجاع بيانات النماذج من sessionStorage
 * 
 * @param {string} formKey - المفتاح الفريد للنموذج في sessionStorage
 * @param {Object} form - نموذج Ant Design
 * @param {Object} dependencies - المتغيرات التي تحتاج إلى حفظها (مثل locationType, rentalType)
 * @returns {Object} - وظائف لحفظ وحذف البيانات
 */
const useFormSessionStorage = (formKey, form, dependencies = {}) => {
  const isInitialMount = useRef(true);
  const savedDependencies = useRef({});

  // استرجاع البيانات المحفوظة عند تحميل الكومبوننت
  useEffect(() => {
    if (isInitialMount.current && typeof window !== 'undefined') {
      try {
        const savedData = sessionStorage.getItem(formKey);
        if (savedData) {
          const parsedData = JSON.parse(savedData);
          
          // استرجاع قيم النموذج
          if (parsedData.formValues && form) {
            // تحويل التواريخ من ISO strings إلى dayjs objects
            const formValues = { ...parsedData.formValues };
            Object.keys(formValues).forEach(key => {
              if (key.includes('DateTime') || key.includes('Date') || key.includes('Time')) {
                if (formValues[key]) {
                  try {
                    const dayjs = require('dayjs');
                    formValues[key] = dayjs(formValues[key]);
                  } catch (error) {
                    console.error('Error parsing date:', error);
                  }
                }
              }
            });
            
            form.setFieldsValue(formValues);
          }

          // استرجاع المتغيرات الإضافية
          savedDependencies.current = parsedData.dependencies || {};
          
          // إرجاع الـ dependencies المحفوظة للاستخدام في الكومبوننت
          if (parsedData.dependencies && Object.keys(parsedData.dependencies).length > 0) {
            // يمكن للكومبوننت استخدام هذه القيم لتحديث state
            Object.keys(parsedData.dependencies).forEach(key => {
              if (window[`set${key.charAt(0).toUpperCase()}${key.slice(1)}`]) {
                window[`set${key.charAt(0).toUpperCase()}${key.slice(1)}`](parsedData.dependencies[key]);
              }
            });
          }
        }
      } catch (error) {
        console.error('Error loading form data from sessionStorage:', error);
      }
      isInitialMount.current = false;
    }
  }, [formKey, form]);

  // حفظ البيانات في sessionStorage عند التغيير
  const saveToSession = (additionalData = {}) => {
    if (typeof window !== 'undefined') {
      try {
        const formValues = form ? form.getFieldsValue() : {};
        
        // تحويل التواريخ إلى ISO strings للحفظ
        const serializedValues = { ...formValues };
        Object.keys(serializedValues).forEach(key => {
          if (serializedValues[key] && typeof serializedValues[key] === 'object' && serializedValues[key].$d) {
            // dayjs object
            serializedValues[key] = serializedValues[key].toISOString();
          }
        });

        const dataToSave = {
          formValues: serializedValues,
          dependencies: { ...dependencies, ...additionalData },
          timestamp: new Date().toISOString()
        };

        sessionStorage.setItem(formKey, JSON.stringify(dataToSave));
        savedDependencies.current = dataToSave.dependencies;
      } catch (error) {
        console.error('Error saving form data to sessionStorage:', error);
      }
    }
  };

  // حذف البيانات من sessionStorage
  const clearSession = () => {
    if (typeof window !== 'undefined') {
      try {
        sessionStorage.removeItem(formKey);
        savedDependencies.current = {};
      } catch (error) {
        console.error('Error clearing form data from sessionStorage:', error);
      }
    }
  };

  // الحصول على البيانات المحفوظة
  const getSavedData = () => {
    if (typeof window !== 'undefined') {
      try {
        const savedData = sessionStorage.getItem(formKey);
        if (savedData) {
          return JSON.parse(savedData);
        }
      } catch (error) {
        console.error('Error getting saved data:', error);
      }
    }
    return null;
  };

  // الحصول على الـ dependencies المحفوظة
  const getSavedDependencies = () => {
    return savedDependencies.current;
  };

  return {
    saveToSession,
    clearSession,
    getSavedData,
    getSavedDependencies,
    isInitialLoad: isInitialMount.current
  };
};

export default useFormSessionStorage;

