'use client';

import { URL } from '@/constants/api';
import { useCallback, useEffect, useState } from 'react';
import { useSelector } from 'react-redux';

export const useFilteredCars = (isFromHome = false) => {
    const [cars, setCars] = useState([]);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    
    const { duration, selectedRange, selectedCarTypes, deliveryType, selectedCityId, selectedBranchId } = useSelector((state) => state.rental);
    const { city } = useSelector((state) => state.shared);
    
    // استخدام المدينة المختارة من الفلتر إذا كانت متوفرة، وإلا استخدم المدينة الافتراضية
    const effectiveCityId = selectedCityId || city?.cityId;

    const fetchFilteredCars = useCallback(async () => {
        try {
            setLoading(true);
            setError(null);

            // إذا لم يتم اختيار مدينة، استخدم البيانات الافتراضية
            if (!effectiveCityId) {
                return;
            }

            // تحويل المدة إلى datingType
            const datingType = duration === "daily" ? 1 : duration === "weekly" ? 2 : duration === "monthly" ? 3 : 1;

            // بناء query parameters
            const queryParams = new URLSearchParams();
            queryParams.append('cityId', effectiveCityId);
            queryParams.append('datingType', datingType);
            queryParams.append('showHome', isFromHome ? 1 : 0);
            queryParams.append('page', '0');
            queryParams.append('size', '1000'); // عدد أكبر من السيارات للعرض

            // إضافة نوع السيارات إذا تم اختيارها
            if (selectedCarTypes && selectedCarTypes.length > 0) {
                selectedCarTypes.forEach(carType => {
                    queryParams.append('typeIds', carType);
                });
            }

            // إضافة الفرع إذا تم اختياره
            if (selectedBranchId) {
                queryParams.append('branchId', selectedBranchId);
            }

            // إضافة التاريخ إذا تم اختياره
            if (selectedRange && selectedRange.length === 2) {
                queryParams.append('startDate', selectedRange[0]);
                queryParams.append('endDate', selectedRange[1]);
            }

            const response = await fetch(URL(`/car/filter?${queryParams.toString()}`), {
                cache: 'no-store'
            });

            if (!response.ok) {
                throw new Error('Failed to fetch cars');
            }

            const data = await response.json();
            setCars(data.content || []);
        } catch (err) {
            // Error fetching filtered cars
            setError(err.message);
            setCars([]);
        } finally {
            setLoading(false);
        }
    }, [effectiveCityId, duration, selectedCarTypes, selectedBranchId, selectedRange]);

    // جلب السيارات عند تغيير الفلتر
    useEffect(() => {
        fetchFilteredCars();
    }, [fetchFilteredCars]);

    // جلب السيارات الافتراضية عند تحميل الصفحة
    useEffect(() => {
        if (!effectiveCityId) {
            fetchDefaultCars();
        }
    }, [effectiveCityId]);

    const fetchDefaultCars = async () => {
        try {
            setLoading(true);
            const response = await fetch(URL('/home/alldata'), {
                cache: 'no-store'
            });
            
            if (!response.ok) {
                throw new Error('Failed to fetch default cars');
            }

            const data = await response.json();
            setCars(data.data?.cars || []);
        } catch (err) {
            // Error fetching default cars
            setError(err.message);
            setCars([]);
        } finally {
            setLoading(false);
        }
    };

    return {
        cars,
        loading,
        error,
        refetch: fetchFilteredCars,
        refetchDefault: fetchDefaultCars
    };
};
