/**
 * Hook لمسح sessionStorage عند عمل refresh للصفحة
 * Hook to clear sessionStorage on page refresh
 */
'use client';

import { useEffect } from 'react';

/**
 * Hook يقوم بمسح sessionStorage عند عمل refresh للصفحة
 * @param {boolean} enabled - تفعيل أو تعطيل المسح التلقائي (default: true)
 * @param {string[]} excludeKeys - مفاتيح معينة لا يتم مسحها (optional)
 */
export const useClearOnRefresh = (enabled = true, excludeKeys = []) => {
  useEffect(() => {
    if (!enabled || typeof window === 'undefined') return;

    // التحقق إذا كانت الصفحة تم تحميلها من الذاكرة المؤقتة (back/forward)
    // أو إذا كان تحميل عادي (refresh/first load)
    const isPageRefresh = () => {
      const navigationType = performance.getEntriesByType('navigation')[0]?.type;
      return navigationType === 'reload';
    };

    // مسح sessionStorage عند refresh
    const clearSessionStorage = () => {
      if (isPageRefresh()) {
        if (excludeKeys.length > 0) {
          // حفظ المفاتيح المستثناة
          const savedData = {};
          excludeKeys.forEach(key => {
            const value = sessionStorage.getItem(key);
            if (value) {
              savedData[key] = value;
            }
          });

          // مسح كل شيء
          sessionStorage.clear();

          // إعادة المفاتيح المستثناة
          Object.keys(savedData).forEach(key => {
            sessionStorage.setItem(key, savedData[key]);
          });
        } else {
          // مسح كل sessionStorage
          sessionStorage.clear();
        }
      }
    };

    // تنفيذ المسح عند تحميل الصفحة
    clearSessionStorage();

    // الاستماع لحدث beforeunload لتحديد نوع الإغلاق
    const handleBeforeUnload = () => {
      // وضع علامة للدلالة على أن المستخدم يغلق الصفحة
      sessionStorage.setItem('__isClosing', 'true');
    };

    window.addEventListener('beforeunload', handleBeforeUnload);

    return () => {
      window.removeEventListener('beforeunload', handleBeforeUnload);
    };
  }, [enabled, excludeKeys]);
};

/**
 * مسح sessionStorage يدوياً
 * Manually clear sessionStorage
 * @param {string[]} excludeKeys - مفاتيح معينة لا يتم مسحها
 */
export const clearSessionStorage = (excludeKeys = []) => {
  if (typeof window === 'undefined') return;

  if (excludeKeys.length > 0) {
    // حفظ المفاتيح المستثناة
    const savedData = {};
    excludeKeys.forEach(key => {
      const value = sessionStorage.getItem(key);
      if (value) {
        savedData[key] = value;
      }
    });

    // مسح كل شيء
    sessionStorage.clear();

    // إعادة المفاتيح المستثناة
    Object.keys(savedData).forEach(key => {
      sessionStorage.setItem(key, savedData[key]);
    });
  } else {
    sessionStorage.clear();
  }
};

export default useClearOnRefresh;

