'use client';

import cookieStorage from '@/utils/cookieStorage';
import { useEffect, useState } from 'react';
import { URL } from '../constants/api';

/**
 * Custom Hook لإدارة بيانات الحجوزات والصفحات والتبويبات
 * يتضمن: جلب البيانات، التنقل بين الصفحات، التبويبات، حالات التحميل
 */
export const useBookingsData = ({ endpoint, showTabs = false, vehicleType = 'car' }) => {
  const [bookings, setBookings] = useState([]);
  const [loading, setLoading] = useState(true);
  const [currentPage, setCurrentPage] = useState(1);
  const [activeTab, setActiveTab] = useState('current');
  const pageSize = 8;

  /**
   * تحديد endpoint المناسب بناءً على التبويبات ونوع المركبة
   */
  const getApiEndpoint = () => {
    if (!showTabs) return endpoint;

    if (vehicleType === 'limousine') {
      return activeTab === 'current'
        ? '/limousine-reservations/current'
        : '/limousine-reservations/previous';
    } else {
      return activeTab === 'current'
        ? '/reservation/get-current-reservations'
        : '/reservation/get-finished-reservations';
    }
  };

  /**
   * جلب بيانات الحجوزات من الـ API
   */
  const fetchBookings = async () => {
    setLoading(true);
    try {
      const token = cookieStorage.getItem('token');
      const apiEndpoint = getApiEndpoint();

      const response = await fetch(URL(apiEndpoint), {
        method: 'GET',
        headers: {
          Authorization: `Bearer ${token}`
        }
      });

      const result = await response.json();

      // معالجة البيانات حسب نوع المركبة
      if (vehicleType === 'limousine') {
        if (activeTab === 'current') {
          setBookings(result);
        } else {
          setBookings(result?.content);
        }
      } else {
        setBookings(Array.isArray(result.data) ? result.data : []);
      }
    } catch (error) {
      // Error fetching reservations
      setBookings([]);
    } finally {
      setLoading(false);
    }
  };

  /**
   * معالج تغيير التبويب
   */
  const handleTabChange = (tab) => {
    setActiveTab(tab);
    setCurrentPage(1); // إعادة تعيين الصفحة الأولى عند تغيير التبويب
  };

  /**
   * الحصول على رسالة الفراغ المناسبة
   */
  const getEmptyMessageKey = (t, emptyKey) => {
    if (showTabs) {
      if (vehicleType === 'limousine') {
        return activeTab === 'current'
          ? t('noCurrentLimousineBookings')
          : t('noPreviousLimousineBookings');
      } else {
        return activeTab === 'current'
          ? t('noCurrentCarBookings')
          : t('noPreviousCarBookings');
      }
    }
    return t(emptyKey);
  };

  /**
   * تحديد البيانات المعروضة حسب الصفحة الحالية
   */
  const getCurrentPageData = () => {
    const startIndex = (currentPage - 1) * pageSize;
    return Array.isArray(bookings) ? bookings.slice(startIndex, startIndex + pageSize) : [];
  };

  /**
   * الحصول على تسمية التبويب
   */
  const getTabLabel = (tab, t) => {
    if (vehicleType === 'limousine') {
      return tab === 'current'
        ? t('currentLimousineBookings')
        : t('previousLimousineBookings');
    } else {
      return tab === 'current'
        ? t('currentCarBookings')
        : t('previousCarBookings');
    }
  };

  /**
   * تحميل البيانات عند تغيير المعاملات
   */
  useEffect(() => {
    fetchBookings();
  }, [endpoint, showTabs, activeTab, vehicleType]);

  return {
    // البيانات
    bookings,
    currentPageData: getCurrentPageData(),

    // حالات التحميل والصفحات
    loading,
    currentPage,
    activeTab,
    pageSize,

    // الدوال
    handleTabChange,
    setCurrentPage,
    fetchBookings,
    getEmptyMessageKey,
    getTabLabel,

    // إحصائيات
    totalBookings: Array.isArray(bookings) ? bookings.length : 0,
    hasBookings: Array.isArray(bookings) && bookings.length > 0,
  };
};

export default useBookingsData;