# دليل نظام حفظ بيانات النماذج في Session Storage

## نظرة عامة

تم تطوير نظام متكامل لحفظ بيانات النماذج في `sessionStorage` بحيث تبقى البيانات محفوظة حتى لو انتقل المستخدم بين الصفحات، ولا تُحذف إلا عند:
- إغلاق المتصفح أو التبويب (Session)
- تعديل البيانات من قبل المستخدم

## النماذج المشمولة

### 1. نماذج الليموزين (Limousine Forms)

#### أ. نموذج الاستقبال من المطار (`LimReceptionPricing`)
- **المفتاح**: `lim_reception_form`
- **البيانات المحفوظة**:
  - نوع الموقع (مطار / أماكن أخرى)
  - مكان الاستقبال والنزول
  - التاريخ والوقت
  - رقم الرحلة

#### ب. نموذج الحجز بالساعة (`LimCarsRentHourDay` - hourly)
- **المفتاح**: `lim_hourly_form`
- **البيانات المحفوظة**:
  - المدينة
  - العنوان (مع الإحداثيات)
  - عدد الساعات
  - التاريخ والوقت

#### ج. نموذج الحجز باليوم (`LimCarsRentHourDay` - daily)
- **المفتاح**: `lim_daily_form`
- **البيانات المحفوظة**:
  - المدينة
  - العنوان (مع الإحداثيات)
  - عدد الأيام
  - التاريخ والوقت

#### د. نوع الإيجار المختار
- **المفتاح**: `limousine_rental_type`
- **القيم الممكنة**: `pickup`, `hourly`, `daily`

### 2. نماذج الإيجار العادي (Rental Forms)

#### أ. نموذج الاستلام من الفرع (`RentalBranchForm`)
- **المفتاح**: `rental_branch_form`
- **البيانات المحفوظة**:
  - المدينة المختارة
  - الفرع المختار
  - نطاق التواريخ (من - إلى)
  - وقت البدء
  - أنواع السيارات المختارة

#### ب. نموذج التوصيل (`RentalDeliveryForm`)
- **المفتاح**: `rental_delivery_form`
- **البيانات المحفوظة**:
  - المدينة المختارة
  - موقع الاستلام (العنوان + الإحداثيات)
  - نطاق التواريخ (من - إلى)
  - وقت البدء
  - أنواع السيارات المختارة

## آلية العمل

### 1. الحفظ التلقائي
- يتم حفظ البيانات تلقائياً عند:
  - تغيير أي قيمة في النموذج
  - اختيار/تغيير المدينة أو الفرع
  - اختيار التواريخ والأوقات
  - كتابة في حقول الإدخال

### 2. الاسترجاع التلقائي
- يتم استرجاع البيانات تلقائياً عند:
  - تحميل الصفحة
  - الانتقال بين التبويبات
  - العودة للصفحة من صفحة أخرى

### 3. معالجة التواريخ
- يتم تحويل كائنات `dayjs` إلى `ISO strings` قبل الحفظ
- يتم تحويل `ISO strings` إلى كائنات `dayjs` عند الاسترجاع

## Hook المستخدم

### `useFormSessionStorage`

```javascript
const { saveToSession, getSavedData, clearSession } = useFormSessionStorage(
  'form_key',
  form,
  { additionalData }
);
```

#### المعاملات:
- `formKey`: المفتاح الفريد للنموذج في sessionStorage
- `form`: نموذج Ant Design
- `dependencies`: البيانات الإضافية التي تحتاج حفظها (state variables)

#### الوظائف المتاحة:
- `saveToSession()`: حفظ البيانات يدوياً
- `getSavedData()`: استرجاع البيانات المحفوظة
- `clearSession()`: حذف البيانات المحفوظة

## مثال على الاستخدام

```javascript
import useFormSessionStorage from '@/hook/useFormSessionStorage';

const MyForm = () => {
  const [form] = Form.useForm();
  const [locationType, setLocationType] = useState('default');
  
  // استخدام الـ hook
  const { saveToSession, getSavedData } = useFormSessionStorage(
    'my_form_key',
    form,
    { locationType }
  );
  
  // استرجاع البيانات عند التحميل
  useEffect(() => {
    const savedData = getSavedData();
    if (savedData) {
      if (savedData.dependencies?.locationType) {
        setLocationType(savedData.dependencies.locationType);
      }
    }
  }, []);
  
  // حفظ عند التغيير
  useEffect(() => {
    saveToSession({ locationType });
  }, [locationType]);
  
  return <Form form={form}>...</Form>;
};
```

## الملاحظات المهمة

1. **حماية البيانات**: جميع عمليات الحفظ والاسترجاع محمية بـ try-catch لتجنب الأخطاء
2. **التحقق من البيئة**: يتم التحقق من `typeof window !== 'undefined'` قبل أي عملية
3. **التوافقية**: النظام متوافق مع جميع المتصفحات الحديثة
4. **الأداء**: يتم الحفظ بشكل فعال دون تأثير على أداء التطبيق

## صيانة النظام

### إضافة نموذج جديد:
1. استخدم الـ hook في الكومبوننت
2. حدد مفتاح فريد للنموذج
3. أضف useEffect لاسترجاع البيانات عند التحميل
4. أضف useEffect لحفظ البيانات عند التغيير

### حذف البيانات المحفوظة:
```javascript
// لنموذج معين
sessionStorage.removeItem('form_key');

// لجميع البيانات
sessionStorage.clear();
```

## الدعم والمساعدة

إذا واجهت أي مشكلة، تحقق من:
1. Console للأخطاء
2. Application > Session Storage في Developer Tools
3. تأكد من صحة مفاتيح الحفظ

---

تم التطوير بواسطة فريق التطوير - 2024

