import { DatePicker, Form } from 'antd';
import arEG from 'antd/es/date-picker/locale/ar_EG';
import enUS from 'antd/es/date-picker/locale/en_US';
import dayjs from 'dayjs';

const BirthdateCalender = ({
  name,
  label,
  placeholder = 'اختر التاريخ',
  size = 'large',
  format = 'DD-MM-YYYY',
  showTime = false,
  localeCode = 'ar',
  defaultValue = null,
  required = true,
  disablePastDates = true, // <-- اختياري للتحكم
  maxDaysFromNow = null,
  rules = [],
  ...props
}) => {
  const locale = localeCode === 'en' ? enUS : arEG;
  const finalRules = rules.length > 0 ? rules : [{ required: required, message: `يرجى اختيار ${label}` }];
  //❗ دالة لتحويل الساعات إلى أيام وتعطيل التواريخ
  const createMaxDaysDisabledDate = (hours) => {
    // التأكد من أن hours رقم صحيح
    const validHours = Number(hours) || 0;
    const maxDays = Math.ceil(validHours / 24); // تحويل الساعات إلى أيام
    const today = dayjs().startOf('day');
    const maxDate = today.add(maxDays, 'day');


    return (current) => {
      if (!current) return false;

      // تعطيل التواريخ السابقة
      if (disablePastDates && current < today) {
        return true;
      }

      // تعطيل التواريخ بعد الحد الأقصى
      if (current > maxDate) {
        return true;
      }

      return false;
    };
  };

  //❗ دالة افتراضية لتعطيل التواريخ السابقة فقط
  const defaultDisabledDate = (current) => {
    if (!disablePastDates) return false;
    return current && current < dayjs().startOf('day');
  };

  //❗ دالة لتعطيل التواريخ المستقبلية فقط (للتاريخ الميلاد)
  const disableFutureDates = (current) => {
    return current && current > dayjs().startOf('day');
  };

  //❗ دالة لتعطيل الأوقات السابقة في اليوم الحالي
  const disabledTime = (current) => {
    if (!current || !showTime) return {};

    const now = dayjs();
    const isToday = current.isSame(now, 'day');

    if (!isToday) return {};

    const currentHour = now.hour();
    const currentMinute = now.minute();

    return {
      disabledHours: () => {
        const hours = [];
        for (let i = 0; i < currentHour; i++) {
          hours.push(i);
        }
        return hours;
      },
      disabledMinutes: (selectedHour) => {
        if (selectedHour === currentHour) {
          const minutes = [];
          for (let i = 0; i < currentMinute; i++) {
            minutes.push(i);
          }
          return minutes;
        }
        return [];
      }
    };
  };

  // تحديد الدالة المناسبة للتعطيل
  const finalDisabledDate = maxDaysFromNow && Number(maxDaysFromNow) > 0
    ? createMaxDaysDisabledDate(maxDaysFromNow)
    : disablePastDates === false 
      ? disableFutureDates  // تعطيل التواريخ المستقبلية فقط (للتاريخ الميلاد)
      : defaultDisabledDate; // تعطيل التواريخ السابقة (الافتراضي)

  return (
    <Form.Item
      name={name}
      label={label}
      rules={finalRules}
    >
      <DatePicker
        placeholder={placeholder}
        size={size}
        format={format}
        showTime={showTime ? { minuteStep: 30 } : false}
        locale={locale}
        defaultValue={defaultValue ? dayjs(defaultValue) : null}
        disabledDate={finalDisabledDate}
        disabledTime={disabledTime}
        className={props.className}

        style={{
          width: '100%',
          direction: localeCode === 'ar' ? 'rtl' : 'ltr',
          textAlign: localeCode === 'ar' ? 'right' : 'left',
        }}

        {...props}
      />
    </Form.Item>
  );
};

export default BirthdateCalender;
