'use client';

import useBranchWorkingHours from "@/hook/useBranchWorkingHours";
import useCarAvailabilityCheck from "@/hook/useCarAvailabilityCheck";
import { getCityCenter, getLimousineLocationErrorMessage, validateLimousineLocation } from "@/utils/limousineLocationValidation";
import { getLocationErrorMessage, validateLocation } from "@/utils/locationValidation";
import { EnvironmentOutlined } from "@ant-design/icons";
import CarUnavailableModal from "@components/ui/CarUnavailableModal";
import {
  GoogleMap,
  Marker,
  StandaloneSearchBox,
  useJsApiLoader,
} from "@react-google-maps/api";
import { Form, Input, message, Modal, Spin } from "antd";
import { useTranslations } from "next-intl";
import { useEffect, useRef, useState } from "react";

const { TextArea } = Input;

export default function DeliveryLocation({
  form,
  addressName = "address",
  latName = "latitude",
  lngName = "longitude",
  label,
  className = "border w-full rounded-md px-4 py-3 border-black transition cursor-pointer flex items-center justify-between",
  onLocationChange,
  // خاص بخدمة الليموزين: تخطي API validation واستخدام validation محلي فقط
  skipApiValidation = false,
  cityId = null, // JEDDAH أو MAKKAH - مطلوب عند استخدام skipApiValidation
  carId = null // معرف السيارة للتحقق من توفرها في الفرع
}) {
  const t = useTranslations();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedAddress, setSelectedAddress] = useState("");
  const [manualAddress, setManualAddress] = useState("");
  const [isValidatingLocation, setIsValidatingLocation] = useState(false);
  const [locationValidation, setLocationValidation] = useState(null);
  const searchBoxRef = useRef(null);
  
  // State لـ Modal عدم توفر السيارة
  const [showCarUnavailableModal, setShowCarUnavailableModal] = useState(false);
  const [unavailableBranchData, setUnavailableBranchData] = useState(null);
  const [unavailableBranchId, setUnavailableBranchId] = useState(null);
  
  // استخدام الـ hooks للتحقق من توفر السيارة وأوقات العمل
  const { checkCarAvailability } = useCarAvailabilityCheck();
  const { fetchWorkingHours, getDayWorkingHours, workingHours } = useBranchWorkingHours();

  // تحديد مركز الخريطة بناءً على المدينة (للخدمات الليموزين) أو الموقع الافتراضي
  const defaultCenter = skipApiValidation && cityId 
    ? getCityCenter(cityId) 
    : { lat: 21.4858, lng: 39.1925 };
  
  const [markerPosition, setMarkerPosition] = useState(defaultCenter);

  const selectedLocation = manualAddress || selectedAddress;

  // تحديث مركز الخريطة عند تغيير المدينة
  useEffect(() => {
    if (skipApiValidation && cityId) {
      const center = getCityCenter(cityId);
      setMarkerPosition(center);
    }
  }, [skipApiValidation, cityId]);

  // مراقبة تغييرات النموذج وتحديث الحالة
  useEffect(() => {
    const existingAddress = form.getFieldValue(addressName);
    const existingLat = form.getFieldValue(latName);
    const existingLng = form.getFieldValue(lngName);

    if (existingAddress) {
      setSelectedAddress(existingAddress);
    }

    if (existingLat && existingLng) {
      setMarkerPosition({
        lat: existingLat,
        lng: existingLng
      });
    }
  }, [form, addressName, latName, lngName]);

  // دالة لتحويل العنوان إلى العربية
  const convertToArabic = (address) => {
    if (!address) return '';

    // قاموس للترجمة
    const translations = {
      'Jeddah': 'جدة',
      'Mecca': 'مكة',
      'Riyadh': 'الرياض',
      'Makkah': 'مكة المكرمة',
      'Al Balad': 'البلد',
      'District': 'حي',
      'Street': 'شارع',
      'Road': 'طريق',
      'Avenue': 'جادة',
      'Saudi Arabia': 'المملكة العربية السعودية',
      'Kingdom': 'المملكة',
      'Arabia': 'العربية السعودية'
    };

    let arabicAddress = address;

    // تطبيق الترجمات
    Object.entries(translations).forEach(([english, arabic]) => {
      const regex = new RegExp(english, 'gi');
      arabicAddress = arabicAddress.replace(regex, arabic);
    });

    return arabicAddress;
  };

  const { isLoaded } = useJsApiLoader({
    googleMapsApiKey: "AIzaSyAtUOb461InzoQoGEVKKVqqLf2NbwSjqdk",
    libraries: ["places"],
  });

  const reverseGeocode = async (lat, lng) => {
    try {
      const res = await fetch(
        `https://maps.googleapis.com/maps/api/geocode/json?latlng=${lat},${lng}&key=AIzaSyAtUOb461InzoQoGEVKKVqqLf2NbwSjqdk&language=ar`
      );
      const data = await res.json();
      console.log('Reverse geocode response:', data);
      
      if (data.status === 'OK' && data.results && data.results.length > 0) {
        const address = data.results[0]?.formatted_address;
        console.log('Formatted address:', address);
        // تحويل العنوان إلى العربية
        const arabicAddress = convertToArabic(address || "");
        console.log('Arabic address:', arabicAddress);
        setSelectedAddress(arabicAddress);
      } else {
        console.log('No address found, using coordinates');
        setSelectedAddress(`الموقع المحدد: ${lat.toFixed(6)}, ${lng.toFixed(6)}`);
      }
    } catch (err) {
      console.error('Reverse geocode error:', err);
      setSelectedAddress(`الموقع المحدد: ${lat.toFixed(6)}, ${lng.toFixed(6)}`);
      message.error(t('location.address_fetch_error'));
    }
  };

  // دالة للتحقق من صحة الموقع
  const validateLocationCoordinates = async (lat, lng) => {
    setIsValidatingLocation(true);
    setLocationValidation(null);
    
    try {
      let validationResult;
      
      // إذا كانت خدمة ليموزين، استخدم validation محلي بدون API
      if (skipApiValidation) {
        // التحقق من وجود المدينة
        if (!cityId) {
          const errorResult = {
            isValid: false,
            city: null,
            message: 'يرجى اختيار المدينة أولاً',
            error: 'NO_CITY_SELECTED'
          };
          setLocationValidation(errorResult);
          message.error(getLimousineLocationErrorMessage(errorResult, 'ar'));
          setIsValidatingLocation(false);
          return errorResult;
        }
        
        // التحقق المحلي للموقع
        validationResult = validateLimousineLocation(lat, lng, cityId);
        setLocationValidation(validationResult);
        
        if (!validationResult.isValid) {
          const errorMessage = getLimousineLocationErrorMessage(validationResult, 'ar');
          message.error(errorMessage);
        } else {
          message.success(t('location.location_validated_successfully'));
        }
      } else {
        // استخدام API validation للخدمات العادية
        validationResult = await validateLocation(lat, lng);
        setLocationValidation(validationResult);
        
        if (!validationResult.isValid) {
          const errorMessage = getLocationErrorMessage(validationResult, 'ar');
          message.error(errorMessage);
        } else {
          // إذا كان الموقع صالح، نتحقق من توفر السيارة في الفرع وأوقات العمل
          const nearestBranch = validationResult.branch;
          
          if (nearestBranch && nearestBranch.branchId) {
            // ✅ 1. التحقق من توفر السيارة في الفرع
            if (carId) {
              const carAvailable = await checkCarAvailability({ 
                carId, 
                branchId: nearestBranch.branchId 
              });
              
              if (!carAvailable) {
                // إظهار Modal عدم توفر السيارة
                setUnavailableBranchData(nearestBranch);
                setUnavailableBranchId(nearestBranch.branchId);
                setShowCarUnavailableModal(true);
                
                const errorResult = {
                  isValid: false,
                  branch: nearestBranch,
                  message: t('validation.CAR_NOT_AVAILABLE_IN_BRANCH'),
                  error: 'CAR_NOT_AVAILABLE'
                };
                setLocationValidation(errorResult);
                setIsValidatingLocation(false);
                return errorResult;
              }
            }
            
            // ✅ 2. التحقق من أوقات عمل الفرع
            await fetchWorkingHours(nearestBranch.branchId);
            
            // إضافة معلومات الفرع إلى النتيجة
            validationResult.branchWorkingHours = workingHours;
          }
          
          message.success(t('location.location_validated_successfully'));
        }
      }
      
      return validationResult;
    } catch (error) {
      console.error('Location validation error:', error);
      const errorResult = {
        isValid: false,
        branch: null,
        message: 'Failed to validate location',
        error: 'NETWORK_ERROR'
      };
      setLocationValidation(errorResult);
      message.error(t('location.validation_error'));
      return errorResult;
    } finally {
      setIsValidatingLocation(false);
    }
  };

  const handleMapClick = async (e) => {
    const lat = e.latLng?.lat();
    const lng = e.latLng?.lng();
    if (lat && lng) {
      setMarkerPosition({ lat, lng });
      reverseGeocode(lat, lng);
      // التحقق من صحة الموقع
      await validateLocationCoordinates(lat, lng);
    }
  };

  const handleOpenModal = () => {
    // قراءة القيم الموجودة في النموذج
    const existingAddress = form.getFieldValue(addressName);
    const existingLat = form.getFieldValue(latName);
    const existingLng = form.getFieldValue(lngName);

    // إذا كان هناك قيم موجودة، استخدمها
    if (existingLat && existingLng) {
      setMarkerPosition({
        lat: existingLat,
        lng: existingLng
      });
    }

    if (existingAddress) {
      setSelectedAddress(existingAddress);
    }

    setIsModalOpen(true);
  };

  const handleOk = async () => {
    // التحقق من وجود موقع محدد
    if (!markerPosition.lat || !markerPosition.lng) {
      message.warning(t('location.choose_location_warning'));
      return;
    }

    // التحقق من صحة الموقع قبل التأكيد
    const validationResult = await validateLocationCoordinates(markerPosition.lat, markerPosition.lng);
    
    if (!validationResult.isValid) {
      // لا نغلق المودال إذا كان الموقع غير صالح
      return;
    }

    // استخدام العنوان المحدد أو إنشاء عنوان افتراضي
    const addressToUse = selectedLocation || `الموقع المحدد: ${markerPosition.lat.toFixed(6)}, ${markerPosition.lng.toFixed(6)}`;

    // ✅ حفظ الإحداثيات فقط (بدون العنوان) في الفورم الرئيسي
    const formValues = {
      [addressName]: addressToUse, // حفظ في حقل منفصل للخريطة فقط
      [latName]: markerPosition.lat,
      [lngName]: markerPosition.lng,
    };
    form.setFieldsValue(formValues);

    // استدعاء callback إذا كان متوفراً
    if (onLocationChange) {
      onLocationChange({
        address: addressToUse,
        latitude: markerPosition.lat,
        longitude: markerPosition.lng,
        validation: validationResult
      });
    }

    setIsModalOpen(false);
  };

  return (
    <div className="w-full">
      {/* ✅ حقل العنوان الرئيسي */}
      <Form.Item name={addressName} rules={[
        { required: true, message: t('required_address') }
      ]}>
        <div
          className={`border w-full rounded-md px-4 py-3 border-black transition cursor-pointer flex items-center justify-between ${className}`}
          onClick={handleOpenModal}
        >
          <span className="text-gray-700 max-sm:!text-[10px]">
            {(() => {
              const address = form.getFieldValue(addressName);
              if (!address) return t('location.click_to_select', { label });

              // عرض أول كلمتين من العنوان
              const words = address.split(' ');
              const firstTwoWords = words.slice(0, 5).join(' ');
              return firstTwoWords || address;

            })()}
          </span>
          <EnvironmentOutlined className="text-xl text-gray-500 max-sm:!hidden" />
        </div>
      </Form.Item>

      {/* ✅ الحقول المخفية لخط العرض والطول */}
      <Form.Item name={latName} hidden>
        <Input type="hidden" />
      </Form.Item>
      <Form.Item name={lngName} hidden>
        <Input type="hidden" />
      </Form.Item>

      {/* ✅ المودال الخاص بالخريطة */}
      <Modal
        title={<h2 className="text-xl font-bold text-center">{label}</h2>}
        open={isModalOpen}
        onOk={handleOk}
        onCancel={() => setIsModalOpen(false)}
        okText={t('location.confirm')}
        cancelText={t('location.cancel')} 
        centered
      >
        {isLoaded ? (
          <div className="space-y-6">
            <div className="relative w-full h-[400px] rounded overflow-hidden">
              <StandaloneSearchBox
                onLoad={(ref) => (searchBoxRef.current = ref)}
                onPlacesChanged={async () => {
                  const places = searchBoxRef.current.getPlaces();
                  if (places.length > 0) {
                    const location = places[0].geometry?.location;
                    if (location) {
                      const lat = location.lat();
                      const lng = location.lng();
                      setMarkerPosition({ lat, lng });
                      reverseGeocode(lat, lng);
                      // التحقق من صحة الموقع
                      await validateLocationCoordinates(lat, lng);
                    }
                  }
                }}
              >
                <input
                  type="text"
                  placeholder={t('location.search_location')}
                  className="absolute z-[1] top-3 right-3 w-72 p-2 rounded shadow bg-white border focus:outline-none"
                />
              </StandaloneSearchBox>

              <GoogleMap
                center={markerPosition}
                zoom={15}
                mapContainerStyle={{ width: "100%", height: "100%" }}
                onClick={handleMapClick}
                options={{
                  mapTypeControl: false,
                  fullscreenControl: false,
                  streetViewControl: false,
                }}
              >
                <Marker
                  position={markerPosition}
                  draggable
                  onDragEnd={async (e) => {
                    const lat = e.latLng?.lat();
                    const lng = e.latLng?.lng();
                    if (lat && lng) {
                      setMarkerPosition({ lat, lng });
                      reverseGeocode(lat, lng);
                      // التحقق من صحة الموقع
                      await validateLocationCoordinates(lat, lng);
                    }
                  }}
                />
              </GoogleMap>

              <button
                onClick={async () => {
                  if (navigator.geolocation) {
                    navigator.geolocation.getCurrentPosition(
                      async (position) => {
                        const { latitude, longitude } = position.coords;
                        const newPosition = { lat: latitude, lng: longitude };
                        setMarkerPosition(newPosition);
                        reverseGeocode(latitude, longitude);
                        // التحقق من صحة الموقع
                        await validateLocationCoordinates(latitude, longitude);
                      },
                      () => {
                        message.warning(t('location.current_location_error'));
                      }
                    );
                  } else {
                    message.warning(t('location.geolocation_not_supported'));
                  }
                }}
                className="absolute top-3 left-3 z-10 bg-white shadow px-4 py-2 rounded font-semibold text-sm hover:bg-gray-100 transition"
              >
                {t('location.current_location')}
              </button>
            </div>

            {/* <Divider className="my-4">أو</Divider>

            <div>
              <h3 className="text-lg font-semibold mb-2">أدخل العنوان يدويًا</h3>
              <TextArea
                rows={4}
                placeholder="اكتب العنوان بالتفصيل"
                value={manualAddress}
                onChange={(e) => setManualAddress(e.target.value)}
              />
            </div> */}

            <div className="text-sm text-gray-700 bg-gray-50 border rounded p-3 mt-4">
              <strong>{t('location.selected_address')}:</strong>
              <div className="mt-2" >
                {(() => {
                  if (!selectedLocation) return t('location.no_address_selected');
                  const arabicAddress = convertToArabic(selectedLocation);
                  // const words = arabicAddress.split(' ').slice(0, 4);
                  const words = arabicAddress.split(' ');
                  // return words.join(' ') + (arabicAddress.split(' ').length > 4 ? '...' : '');
                  return words.join(' ');

                })()}
              </div>
              
              {/* Validation Status */}
              {isValidatingLocation && (
                <div className="mt-3 flex items-center gap-2">
                  <Spin size="small" />
                  <span className="text-blue-600">{t('location.validating_location')}</span>
                </div>
              )}
              
              {locationValidation && !isValidatingLocation && (
                <div className={`mt-3 p-2 rounded text-sm ${
                  locationValidation.isValid 
                    ? 'bg-green-100 text-green-800 border border-green-200' 
                    : 'bg-red-100 text-red-800 border border-red-200'
                }`}>
                  <div className="flex items-center gap-2">
                    <span className={`w-2 h-2 rounded-full ${
                      locationValidation.isValid ? 'bg-green-500' : 'bg-red-500'
                    }`}></span>
                    <span>
                      {locationValidation.isValid 
                        ? t('location.location_valid') 
                        : skipApiValidation 
                          ? getLimousineLocationErrorMessage(locationValidation, 'ar')
                          : getLocationErrorMessage(locationValidation, 'ar')
                      }
                    </span>
                  </div>
                  {/* عرض معلومات الفرع للخدمات العادية */}
                  {!skipApiValidation && locationValidation.branch && (
                    <div className="mt-2 space-y-1 flex justify-between">
                      <div className="text-xs">
                        <strong>{t('location.nearest_branch')}:</strong> {locationValidation.branch.branchArName || locationValidation.branch.branchName}
                      </div>
                      {/* عرض أوقات عمل الفرع */}
                      {locationValidation.branch.workingHoures && (
                        <div className="text-xs">
                          <strong>ساعات العمل:</strong> {locationValidation.branch.workingHoures}
                        </div>
                      )} 
                    </div>
                  )}
                  {/* عرض معلومات المدينة للخدمات الليموزين */}
                  {skipApiValidation && locationValidation.city && (
                    <div className="mt-1 text-xs">
                      <strong>المدينة:</strong> {locationValidation.city.name}
                      {locationValidation.distance && (
                        <span className="mr-2">
                          • المسافة من المركز: {locationValidation.distance} كم
                        </span>
                      )}
                    </div>
                  )}
                </div>
              )}
            </div>
          </div>
        ) : (
          <div>{t('location.loading_map')}</div>
        )}
      </Modal>
      
      {/* Modal عدم توفر السيارة */}
      <CarUnavailableModal
        open={showCarUnavailableModal}
        branchData={unavailableBranchData}
        branchId={unavailableBranchId}
        onClose={() => setShowCarUnavailableModal(false)}
      />
    </div>
  );
}
