"use client";
import { Button, Card, Col, Row, Typography } from 'antd';
import dayjs from 'dayjs';
import { useTranslations } from 'next-intl';
import { useEffect, useRef, useState } from 'react';
import './TimeSlotSelector.css';

const { Title, Text } = Typography;

const TimeSlotSelector = ({
    onTimeSelect = null,
    className = "",
    className1 = "",
    startHour = 9, // بداية من 9 صباحاً
    endHour = 18,  // نهاية في 6 مساءً
    interval = 30, // فاصل 30 دقيقة 
    showTimeGrid = true,
    selectedTime: initialSelectedTime = null,
    workingHours = null, // مواعيد العمل للفرع
    selectedDate = null, // التاريخ المحدد لتحديد اليوم
    startDate = null, // تاريخ البداية (للتحقق من end date)
    startTime = null, // وقت البداية (للتحقق من end time)
    maxHoursFromNow = null, // الحد الأقصى للساعات من الآن (NUM_HOURS_FOR_START_RESERVATION_FROM_BRANCH)
    title = "أختر الوقت",
    isLocation = false // تحديد إذا كان اختيار location أم branch

}) => {
    const t = useTranslations();
    const [selectedTime, setSelectedTime] = useState(initialSelectedTime);

    // مراقبة تغييرات الـ selectedTime prop من الخارج
    useEffect(() => {
        setSelectedTime(initialSelectedTime);
    }, [initialSelectedTime]);
    const [timeSlots, setTimeSlots] = useState([]);
    const scrollContainerRef = useRef(null);

    // دالة لتحديد اليوم من التاريخ
    const getDayName = (date) => {
        if (!date) return null;

        try {
            // التأكد من أن التاريخ صالح
            const dayjsDate = dayjs(date);
            if (!dayjsDate.isValid()) {
                return null;
            }

            const dayNames = ['sun', 'mon', 'tues', 'wed', 'thurs', 'fri', 'sat'];
            const dayIndex = dayjsDate.day();
            return dayNames[dayIndex];
        } catch (error) {
            return null;
        }
    };

    // دالة للحصول على اسم اليوم بالصيغة المطلوبة للـ API (SUNDAY, MONDAY, etc.)
    const getDayNameForAPI = (date) => {
        if (!date) return null;

        try {
            const dayjsDate = dayjs(date);
            if (!dayjsDate.isValid()) {
                return null;
            }

            const dayNames = ['SUNDAY', 'MONDAY', 'TUESDAY', 'WEDNESDAY', 'THURSDAY', 'FRIDAY', 'SATURDAY'];
            const dayIndex = dayjsDate.day();
            return dayNames[dayIndex];
        } catch (error) {
            return null;
        }
    };

    // دالة للتحقق إذا كان الوقت في فترة بريك
    const isTimeInBreak = (hour, minute, date) => {
        if (!workingHours || !workingHours.breaks || !date) {
            return false;
        }

        const dayName = getDayNameForAPI(date);
        if (!dayName) return false;

        // الحصول على جميع فترات البريك لهذا اليوم
        const dayBreaks = workingHours.breaks.filter(breakItem => breakItem.dayOfWeek === dayName);

        // التحقق إذا كان الوقت الحالي في أي فترة بريك
        const currentTimeInMinutes = hour * 60 + minute;

        for (const breakItem of dayBreaks) {
            // تحويل startTime و endTime إلى دقائق
            const startTimeParts = breakItem.startTime.split(':');
            const startHour = parseInt(startTimeParts[0]);
            const startMinute = parseInt(startTimeParts[1]);
            const startTimeInMinutes = startHour * 60 + startMinute;

            const endTimeParts = breakItem.endTime.split(':');
            const endHour = parseInt(endTimeParts[0]);
            const endMinute = parseInt(endTimeParts[1]);
            const endTimeInMinutes = endHour * 60 + endMinute;

            // التحقق إذا كان الوقت الحالي بين startTime و endTime (شامل وقت النهاية)
            if (currentTimeInMinutes >= startTimeInMinutes && currentTimeInMinutes <= endTimeInMinutes) {
                return true;
            }
        }

        return false;
    };

    // إنشاء فترات زمنية لجميع الساعات (24 ساعة)
    useEffect(() => {
        const slots = [];
        const now = dayjs();
        // استخدام التاريخ الممرر أو التاريخ الحالي إذا لم يكن موجوداً
        const effectiveDate = selectedDate ? dayjs(selectedDate) : now;
        const currentDay = getDayName(effectiveDate);
        const isToday = effectiveDate.isSame(now, 'day');

        // التحقق من أن التاريخ المحدد هو نفس تاريخ البداية
        const isSameAsStartDate = startDate && selectedDate && 
            dayjs(selectedDate).isSame(dayjs(startDate), 'day');

        // إنشاء جميع الأوقات من 00:00 إلى 23:59
        for (let hour = 0; hour < 24; hour++) {
            for (let minute = 0; minute < 60; minute += interval) {
                if (hour === 23 && minute >= 60) break;

                const time = dayjs().hour(hour).minute(minute);
                const timeString = time.format('h:mm A');
                const timeValue = time.format('HH:mm');
                
                // تحديد AM/PM بناءً على hour مباشرة (بدون الاعتماد على dayjs)
                const isAM = hour === 0 || (hour >= 1 && hour < 12);
                const isPM = hour === 12 || (hour >= 13 && hour < 24);
                
                // تحديد AM/PM بشكل صحيح - منطق واضح ومحدد
                let timeDisplay;
                if (hour === 0) {
                    // 12:00 AM - منتصف الليل
                    timeDisplay = timeString.replace('AM', t('time_am'));
                } else if (hour >= 1 && hour <= 11) {
                    // 1:00 AM - 11:59 AM - صباحاً
                    timeDisplay = timeString.replace('AM', t('time_am'));
                } else if (hour === 12) {
                    // 12:00 PM - ظهراً (تأكد من أنه PM وليس AM)
                    timeDisplay = timeString.replace(/AM|PM/g, t('time_pm'));
                } else {
                    // 1:00 PM - 11:59 PM - مساءً
                    timeDisplay = timeString.replace('PM', t('time_pm'));
                }
                
                // التحقق الإضافي للتأكد من صحة النتيجة
                if (hour === 12 && timeDisplay.includes(t('time_am'))) {
                    timeDisplay = timeDisplay.replace(t('time_am'), t('time_pm'));
                }

                // التحقق من أن الوقت بعد الوقت الحالي إذا كان اليوم
                let isAfterCurrentTime = true;
                if (isToday) { // التحقق إذا كان اليوم (سواء location أو branch)
                    const currentTimeInMinutes = now.hour() * 60 + now.minute();
                    const slotTimeInMinutes = hour * 60 + minute;
                    isAfterCurrentTime = slotTimeInMinutes > currentTimeInMinutes;
                }

                // التحقق من أن الوقت بعد وقت البداية إذا كان نفس تاريخ البداية
                let isAfterStartTime = true;
                if (!isLocation && isSameAsStartDate && startTime) { // التحقق فقط إذا كان branch
                    const startTimeValue = startTime.value || startTime;
                    const slotTimeInMinutes = hour * 60 + minute;
                    
                    // تحويل وقت البداية إلى دقائق
                    const startTimeParts = startTimeValue.split(':');
                    const startHourValue = parseInt(startTimeParts[0]);
                    const startMinuteValue = parseInt(startTimeParts[1]);
                    const startTimeInMinutes = startHourValue * 60 + startMinuteValue;
                    
                    // يجب أن يكون الوقت بعد وقت البداية
                    isAfterStartTime = slotTimeInMinutes > startTimeInMinutes;
                }

                // التحقق من الحد الأقصى للساعات من الآن (NUM_HOURS_FOR_START_RESERVATION_FROM_BRANCH)
                // إذا كان NUM_HOURS_FOR_START_RESERVATION_FROM_BRANCH = 24 ساعة
                // والوقت الحالي 6:12، فسيتم السماح بالحجز حتى 6:12 غداً فقط
                let isWithinMaxHours = true;
                if (maxHoursFromNow && selectedDate) {
                    const now = dayjs();
                    const maxAllowedDateTime = now.add(maxHoursFromNow, 'hour');
                    const selectedDateTime = dayjs(selectedDate).hour(hour).minute(minute);
                    
                    // التحقق من أن التاريخ المحدد هو نفس تاريخ الحد الأقصى (بكره)
                    const isMaxDate = selectedDate && dayjs(selectedDate).isSame(maxAllowedDateTime, 'day');
                    
                    if (isMaxDate) {
                        // إذا كان التاريخ هو نفس تاريخ الحد الأقصى (بكره)
                        // فآخر وقت للحجز هو نفس الوقت الحالي
                        const currentTimeInMinutes = now.hour() * 60 + now.minute();
                        const slotTimeInMinutes = hour * 60 + minute;
                        isWithinMaxHours = slotTimeInMinutes <= currentTimeInMinutes;
                    } else {
                        // للتواريخ الأخرى، استخدم المنطق العادي
                        isWithinMaxHours = selectedDateTime.isBefore(maxAllowedDateTime);
                    }
                }

                // تحديد ما إذا كان الوقت داخل ساعات العمل
                let isWorkingHours = false;

                if (workingHours && currentDay) {
                    // استخدام مواعيد العمل من API
                    const dayKey = currentDay.toLowerCase();
                    const isOpen = workingHours[dayKey] > 0;

                    if (isOpen) {
                        const openHour = workingHours[`${dayKey}OpenHour`];
                        const openMin = workingHours[`${dayKey}OpenMin`];
                        const closeHour = workingHours[`${dayKey}CloseHour`];
                        const closeMin = workingHours[`${dayKey}CloseMin`];

                        // التأكد من وجود قيم صحيحة
                        if (openHour !== undefined && openMin !== undefined &&
                            closeHour !== undefined && closeMin !== undefined) {
                            const currentTimeInMinutes = hour * 60 + minute;
                            const openTimeInMinutes = openHour * 60 + openMin;
                            const closeTimeInMinutes = closeHour * 60 + closeMin;

                            isWorkingHours = currentTimeInMinutes >= openTimeInMinutes && currentTimeInMinutes < closeTimeInMinutes;

                            // التحقق إذا كان الوقت في فترة بريك
                            if (isWorkingHours) {
                                const inBreak = isTimeInBreak(hour, minute, effectiveDate);
                                if (inBreak) {
                                    isWorkingHours = false; // إخفاء الوقت إذا كان في فترة بريك
                                }
                            }
                        }
                        
                    }
                } else {
                    // استخدام المواعيد الافتراضية - مفتوح 24 ساعة
                    isWorkingHours = true;
                }

                // تحديد ما إذا كان الوقت متاح
                // إذا كان location:
                //   - كل الأوقات متاحة باستثناء الأوقات السابقة للوقت الحالي في حالة اليوم
                //   - مع تطبيق validation الحد الأقصى للساعات (maxHoursFromNow)
                // إذا كان branch:
                //   - التحقق من ساعات العمل، الوقت الحالي، وقت البداية، والحد الأقصى للساعات
                let isAvailable;
                if (isLocation) {
                    // إذا كان location، الأوقات متاحة باستثناء الأوقات السابقة للوقت الحالي في اليوم
                    // مع التحقق من الحد الأقصى للساعات
                    isAvailable = isAfterCurrentTime && isWithinMaxHours;
                } else {
                    // إذا كان branch، التحقق الكامل
                    isAvailable = isWorkingHours && isAfterCurrentTime && isAfterStartTime && isWithinMaxHours;
                }

                slots.push({
                    value: timeValue,
                    label: timeDisplay,
                    hour,
                    minute,
                    isAM: isAM,
                    isWorkingHours: isAvailable
                });
            }
        }
        setTimeSlots(slots);
    }, [startHour, endHour, interval, workingHours, selectedDate, startDate, startTime, maxHoursFromNow, isLocation]);

    // تمرير الـ scroll إلى بداية الوقت المتاح (startHour) عند تحميل المكون
    useEffect(() => {
        if (scrollContainerRef.current && timeSlots.length > 0) {
            // البحث عن أول وقت متاح (startHour)
            const firstAvailableSlotIndex = timeSlots.findIndex(slot => slot.isWorkingHours);

            if (firstAvailableSlotIndex !== -1) {
                // حساب عدد الصفوف قبل أول وقت متاح
                const columns = 4; // span={6} = 4 أعمدة
                const rowsBeforeFirstAvailable = Math.floor(firstAvailableSlotIndex / columns);

                // حساب ارتفاع كل صف (زر + فاصل)
                const buttonHeight = 40; // h-[40px]
                const gap = 12; // gutter={[12, 12]}
                const rowHeight = buttonHeight + gap;

                // تمرير الـ scroll إلى بداية الصف الذي يحتوي على أول وقت متاح
                const scrollTop = rowsBeforeFirstAvailable * rowHeight;

                // إضافة تأخير بسيط لضمان عمل الـ scroll
                setTimeout(() => {
                    if (scrollContainerRef.current) {
                        scrollContainerRef.current.scrollTop = scrollTop;
                    }
                }, 100);
            }
        }
    }, [timeSlots]);

    // معالجة اختيار الوقت
    const handleTimeSelect = (timeSlot) => {
        setSelectedTime(timeSlot);

        if (onTimeSelect) {
            onTimeSelect(timeSlot);
        }
    };
    return (
        <div className={`time-slot-selector  mt-[16px] ${className}`}>
            {/* شبكة اختيار الوقت */}
            {showTimeGrid && (
                <Card className={`shadow-sm border-0 [&_.ant-card-body]:p-0 [&_.ant-card-body]:px-2 `}>
                    <div className="mb-[16px] px-[16px] ">
                        <span className="font-cairo font-bold text-[18px] leading-[100%] tracking-[0] text-[#162A2B] text-right align-middle">
                            {title}
                        </span>
                    </div>
                    {/* شبكة الأوقات - 4 أعمدة مع scroll */}
                    <div
                        ref={scrollContainerRef}
                        className={`h-80 overflow-y-auto p-2 border border-gray-100 rounded-sm bg-gray-50 custom-scrollbar ${className1}`}
                    >
                        <Row gutter={[12, 12]} className='px-[16px]'>
                            {timeSlots.map((slot) => {
                                const isSelected = selectedTime?.value === slot.value;
                                const isAvailable = slot.isWorkingHours; // متاح فقط داخل ساعات العمل

                                return (
                                    <Col key={slot.value} span={6}>
                                        <Button
                                            onClick={() => isAvailable && handleTimeSelect(slot)}
                                            disabled={!isAvailable}
                                            className={`
                                                w-full h-[40px] text-sm font-medium transition-all duration-200 !rounded-[4px]
                                                ${isSelected
                                                    ? '!bg-[#FBF7ED] text-[18px] !font-bold !text-[#37474F] !border-secondary border-solid'
                                                    : isAvailable
                                                        ? 'bg-white text-gray-700 border-gray-300 hover:bg-blue-50'
                                                        : 'bg-gray-100 text-[#979797] border-gray-200 cursor-not-allowed'
                                                } 
                                            `}
                                        >
                                            <div className="flex flex-col items-center">
                                                <span>{slot.label}</span>
                                                <span className="text-xs opacity-80">
                                                </span>
                                            </div>
                                        </Button>
                                    </Col>
                                );
                            })}
                        </Row>
                    </div>
                </Card>
            )}
        </div>
    );
};

export default TimeSlotSelector;
