"use client";
import { Button, Card, Col, Row, Typography } from 'antd';
import dayjs from 'dayjs';
import { useTranslations } from 'next-intl';
import { useEffect, useRef, useState } from 'react';
import './TimeSlotSelector.css';

const { Title, Text } = Typography;

const TimeSlotSelector = ({
    onTimeSelect = null,
    className = "",
    className1 = "",
    startHour = 9, // بداية من 9 صباحاً
    endHour = 18,  // نهاية في 6 مساءً
    interval = 30, // فاصل 30 دقيقة 
    showTimeGrid = true,
    selectedTime: initialSelectedTime = null,
    workingHours = null, // مواعيد العمل للفرع
    selectedDate = null, // التاريخ المحدد لتحديد اليوم
    title = "أختر الوقت",

}) => {
    const t = useTranslations();
    const [selectedTime, setSelectedTime] = useState(initialSelectedTime);

    // مراقبة تغييرات الـ selectedTime prop من الخارج
    useEffect(() => {
        setSelectedTime(initialSelectedTime);
    }, [initialSelectedTime]);
    const [timeSlots, setTimeSlots] = useState([]);
    const scrollContainerRef = useRef(null);

    // دالة لتحديد اليوم من التاريخ
    const getDayName = (date) => {
        if (!date) return null;

        try {
            // التأكد من أن التاريخ صالح
            const dayjsDate = dayjs(date);
            if (!dayjsDate.isValid()) {
                return null;
            }

            const dayNames = ['sun', 'mon', 'tues', 'wed', 'thurs', 'fri', 'sat'];
            const dayIndex = dayjsDate.day();
            return dayNames[dayIndex];
        } catch (error) {
            return null;
        }
    };

    // إنشاء فترات زمنية لجميع الساعات (24 ساعة)
    useEffect(() => {
        const slots = [];
        const now = dayjs();
        // استخدام التاريخ الممرر أو التاريخ الحالي إذا لم يكن موجوداً
        const effectiveDate = selectedDate || now;
        const currentDay = getDayName(effectiveDate);
        const isToday = effectiveDate.isSame(now, 'day');


        // إنشاء جميع الأوقات من 00:00 إلى 23:59
        for (let hour = 0; hour < 24; hour++) {
            for (let minute = 0; minute < 60; minute += interval) {
                if (hour === 23 && minute >= 60) break;

                const time = dayjs().hour(hour).minute(minute);
                const timeString = time.format('h:mm A');
                const timeValue = time.format('HH:mm');
                
                // تحديد AM/PM بشكل صحيح
                const isAM = hour == 0 || (hour >= 1 && hour < 12);
                
                // إنشاء نص الوقت مع الترجمة
                const timeDisplay = timeString.replace('AM', t('time_am')).replace('PM', t('time_pm'));

                // التحقق من أن الوقت بعد الوقت الحالي إذا كان اليوم
                let isAfterCurrentTime = true;
                if (isToday) {
                    const currentTimeInMinutes = now.hour() * 60 + now.minute();
                    const slotTimeInMinutes = hour * 60 + minute;
                    isAfterCurrentTime = slotTimeInMinutes > currentTimeInMinutes;
                }

                // تحديد ما إذا كان الوقت داخل ساعات العمل
                let isWorkingHours = false;

                if (workingHours && currentDay) {
                    // استخدام مواعيد العمل من API
                    const dayKey = currentDay.toLowerCase();
                    const isOpen = workingHours[dayKey] > 0;

                    if (isOpen) {
                        const openHour = workingHours[`${dayKey}OpenHour`];
                        const openMin = workingHours[`${dayKey}OpenMin`];
                        const closeHour = workingHours[`${dayKey}CloseHour`];
                        const closeMin = workingHours[`${dayKey}CloseMin`];

                        // التأكد من وجود قيم صحيحة
                        if (openHour !== undefined && openMin !== undefined &&
                            closeHour !== undefined && closeMin !== undefined) {
                            const currentTimeInMinutes = hour * 60 + minute;
                            const openTimeInMinutes = openHour * 60 + openMin;
                            const closeTimeInMinutes = closeHour * 60 + closeMin;

                            isWorkingHours = currentTimeInMinutes >= openTimeInMinutes && currentTimeInMinutes < closeTimeInMinutes;
                        }
                        
                    }
                } else {
                    // استخدام المواعيد الافتراضية - مفتوح 24 ساعة
                    isWorkingHours = true;
                }

                // الوقت متاح فقط إذا كان داخل ساعات العمل وبعد الوقت الحالي (إذا كان اليوم)
                const isAvailable = isWorkingHours && isAfterCurrentTime;

                slots.push({
                    value: timeValue,
                    label: timeDisplay,
                    hour,
                    minute,
                    isAM: isAM,
                    isWorkingHours: isAvailable
                });
            }
        }
        setTimeSlots(slots);
    }, [startHour, endHour, interval, workingHours, selectedDate]);

    // تمرير الـ scroll إلى بداية الوقت المتاح (startHour) عند تحميل المكون
    useEffect(() => {
        if (scrollContainerRef.current && timeSlots.length > 0) {
            // البحث عن أول وقت متاح (startHour)
            const firstAvailableSlotIndex = timeSlots.findIndex(slot => slot.isWorkingHours);

            if (firstAvailableSlotIndex !== -1) {
                // حساب عدد الصفوف قبل أول وقت متاح
                const columns = 4; // span={6} = 4 أعمدة
                const rowsBeforeFirstAvailable = Math.floor(firstAvailableSlotIndex / columns);

                // حساب ارتفاع كل صف (زر + فاصل)
                const buttonHeight = 40; // h-[40px]
                const gap = 12; // gutter={[12, 12]}
                const rowHeight = buttonHeight + gap;

                // تمرير الـ scroll إلى بداية الصف الذي يحتوي على أول وقت متاح
                const scrollTop = rowsBeforeFirstAvailable * rowHeight;

                // إضافة تأخير بسيط لضمان عمل الـ scroll
                setTimeout(() => {
                    if (scrollContainerRef.current) {
                        scrollContainerRef.current.scrollTop = scrollTop;
                    }
                }, 100);
            }
        }
    }, [timeSlots]);

    // معالجة اختيار الوقت
    const handleTimeSelect = (timeSlot) => {
        setSelectedTime(timeSlot);

        if (onTimeSelect) {
            onTimeSelect(timeSlot);
        }
    };
    return (
        <div className={`time-slot-selector  mt-[16px] ${className}`}>
            {/* شبكة اختيار الوقت */}
            {showTimeGrid && (
                <Card className={`shadow-sm border-0 [&_.ant-card-body]:p-0 [&_.ant-card-body]:px-2 `}>
                    <div className="mb-[16px] px-[16px] ">
                        <span className="font-cairo font-bold text-[18px] leading-[100%] tracking-[0] text-[#162A2B] text-right align-middle">
                            {title}
                        </span>
                    </div>
                    {/* شبكة الأوقات - 4 أعمدة مع scroll */}
                    <div
                        ref={scrollContainerRef}
                        className={`h-80 overflow-y-auto p-2 border border-gray-100 rounded-sm bg-gray-50 custom-scrollbar ${className1}`}
                    >
                        <Row gutter={[12, 12]} className='px-[16px]'>
                            {timeSlots.map((slot) => {
                                const isSelected = selectedTime?.value === slot.value;
                                const isAvailable = slot.isWorkingHours; // متاح فقط داخل ساعات العمل

                                return (
                                    <Col key={slot.value} span={6}>
                                        <Button
                                            onClick={() => isAvailable && handleTimeSelect(slot)}
                                            disabled={!isAvailable}
                                            className={`
                                                w-full h-[40px] text-sm font-medium transition-all duration-200 !rounded-[4px]
                                                ${isSelected
                                                    ? '!bg-[#FBF7ED] text-[18px] !font-bold !text-[#37474F] !border-secondary border-solid'
                                                    : isAvailable
                                                        ? 'bg-white text-gray-700 border-gray-300 hover:bg-blue-50'
                                                        : 'bg-gray-100 text-[#979797] border-gray-200 cursor-not-allowed'
                                                } 
                                            `}
                                        >
                                            <div className="flex flex-col items-center">
                                                <span>{slot.label}</span>
                                                <span className="text-xs opacity-80">
                                                </span>
                                            </div>
                                        </Button>
                                    </Col>
                                );
                            })}
                        </Row>
                    </div>
                </Card>
            )}
        </div>
    );
};

export default TimeSlotSelector;
