'use client';

import { formatCurrency } from '@/utils/format';
import { getCorrectTaxAmount } from '@/utils/taxCalculator';
import Riyal from '@components/svg/Riyal';
import { Button, Divider, Modal, Space } from 'antd';
import { useLocale, useTranslations } from 'next-intl';

const PriceConfirmationModal = ({ 
  isOpen, 
  onClose, 
  onConfirm,
  priceDetails, 
  carData, 
  loading = false,
  type = 'rental', // 'rental' or 'limousine'
  bookingDetails = null // For limousine booking details
}) => {
  const t = useTranslations();
  const locale = useLocale();

  // Define fields based on type
  const rentalFields = [
    { key: 'extraKmCost', label: t('extraKmCost') },
    { key: 'anotherBranchPrice', label: t('branch_change_fee') },
    { key: 'deliveryFee', label: t('delivery_fee') },
    { key: 'extraServicesCost', label: t('extra_services_cost') },
    { key: 'insurence', label: t('insurance_cost') },
    { key: 'membershipDiscount', label: t('membership_discount') },
    { key: 'totalPoints', label: t('points_discount') },
    { key: 'totalPromoValue', label: t('promo_code_discount') },
    { key: 'tax', label: `${t('tax_amount')} 15%` },
  ];

  const limousineFields = [
    { key: 'additionalServicesNetPrice', label: t('extra_services_cost') },
    { key: 'extraKmPrice', label: t('extra_km_cost') },
    { key: 'membershipDiscount', label: t('membership_discount') },
    { key: 'pointsDiscount', label: t('points_discount') },
    { key: 'promoDiscount', label: t('promo_code_discount') },
    { key: 'extraHoursNetPrice', label: t('extra_hours_price') },
    { key: 'vat', label: `${t('tax_amount')} 15%` },
  ];

  const fields = type === 'limousine' ? limousineFields : rentalFields;
  const basePrice = type === 'limousine' ? priceDetails?.grossPrice : priceDetails?.finalPriceBeforeDiscount;
  const finalPrice = type === 'limousine' ? priceDetails?.finalPrice : priceDetails?.finalTotalPrice;
  
  // Get correct tax amount using utility function
  const calculatedTax = getCorrectTaxAmount(priceDetails, basePrice, type);

  return (
    <Modal
      open={isOpen}
      onCancel={onClose}
      footer={null}
      width={500}
      centered
      closable={true}
      maskClosable={false}
      title={
        <div className="text-center">
          <h2 className="text-xl font-bold text-center text-main mb-0">
            {t('confirm_booking_price')}
          </h2>
        </div>
      }
    >
      <div className="py-4">
        {/* Car Information */}
        {carData && (
          <div className="mb-4 p-3 bg-gray-50 rounded-lg">
            <div className="flex items-center gap-3">
              {carData?.image && (
                <img 
                  src={carData?.image} 
                  alt={carData.name} 
                  className="w-16 h-12 object-cover rounded"
                />
              )}
              <div>
                <h3 className="font-semibold">{carData.name}</h3>
                {carData.model && <p className="text-sm text-gray-600">{carData.modelName} {carData.year}</p>}
              </div>
            </div>
          </div>
        )}

        {/* Limousine Booking Details */}
        {type === 'limousine' && bookingDetails && (
          <div className="mb-4 space-y-3">
            {/* Service Type */}
            <div className="p-3 bg-blue-50 rounded-lg">
              <div className="flex justify-between items-start">
                <span className="text-sm font-medium text-gray-700">{t('service_type')}:</span>
                <span className="text-sm font-semibold text-main">
                  {bookingDetails.typeLim === "Pickup_Airport" && t('airport_pickup')}
                  {bookingDetails.typeLim === "Pickup_OtherPlaces" && t('other_places_pickup')}
                  {bookingDetails.typeLim === "hourly" && t('hourly_rental')}
                  {bookingDetails.typeLim === "daily" && t('daily_rental')}
                </span>
              </div>
            </div>

            {/* Date and Time */}
            {bookingDetails.receptionDateTime && (
              <div className="flex justify-between items-center p-2 border-b">
                <span className="text-sm text-gray-600">{t('booking_date_time')}:</span>
                <span className="text-sm font-medium">
                  {new Date(bookingDetails.receptionDateTime).toLocaleString(locale === 'ar' ? 'ar-EG' : 'en-EG', {
                    year: 'numeric',
                    month: 'long',
                    day: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                  })}
                </span>
              </div>
            )}

            {/* Pickup Location - For Airport/Other Places */}
            {(bookingDetails.typeLim === "Pickup_Airport" || bookingDetails.typeLim === "Pickup_OtherPlaces") && (
              <>
                {bookingDetails.from && (
                  <div className="flex justify-between items-center p-2 border-b">
                    <span className="text-sm text-gray-600">{t('pickup_place')}:</span>
                    <span className="text-sm font-medium">{t('locations.' + bookingDetails.from)}</span>
                  </div>
                )}
                
                {bookingDetails.to && (
                  <div className="flex justify-between items-center p-2 border-b">
                    <span className="text-sm text-gray-600">{t('dropoff_place')}:</span>
                    <span className="text-sm font-medium">{t('locations.' + bookingDetails.to)}</span>
                  </div>
                )}

                {bookingDetails.flightNumber && (
                  <div className="flex justify-between items-center p-2 border-b">
                    <span className="text-sm text-gray-600">{t('flight_number')}:</span>
                    <span className="text-sm font-medium">#{bookingDetails.flightNumber}</span>
                  </div>
                )}
              </>
            )}

            {/* Duration - For Hourly/Daily */}
            {bookingDetails.typeLim === "hourly" && bookingDetails.hours && (
              <div className="flex justify-between items-center p-2 border-b">
                <span className="text-sm text-gray-600">{t('rental_duration')}:</span>
                <span className="text-sm font-medium">
                  {bookingDetails.hours} {bookingDetails.hours === 1 ? t('hour') : t('hours')}
                </span>
              </div>
            )}

            {bookingDetails.typeLim === "daily" && bookingDetails.days && (
              <div className="flex justify-between items-center p-2 border-b">
                <span className="text-sm text-gray-600">{t('rental_duration')}:</span>
                <span className="text-sm font-medium">
                  {bookingDetails.days} {bookingDetails.days === 1 ? t('day') : t('days')}
                </span>
              </div>
            )}

            {/* Address if available */}
            {bookingDetails.address && (
              <div className="flex justify-between items-start p-2 border-b">
                <span className="text-sm text-gray-600">{t('detailed_address')}:</span>
                <span className="text-sm font-medium text-right max-w-[60%]">{bookingDetails.address}</span>
              </div>
            )}
          </div>
        )}

        <Divider />

        {/* Total Price */}
        <div className="flex justify-between items-center text-lg font-bold mb-2">
          <span className="text-main">{t('total_amount')}</span>
          <div className="flex items-center gap-2 text-secondary">
            <span>{formatCurrency(finalPrice)}</span>
            <Riyal color="#CE931A" />
          </div>
        </div>

        <div className="text-center text-sm text-gray-500 mb-6">
          {t('vat_included')}
        </div>

        {/* Action Buttons */}
        <Space direction="vertical" style={{ width: '100%' }} size="middle">
          <Button
            type="primary"
            size="large"
            block
            onClick={onConfirm}
            loading={loading}
            className="bg-main border-main hover:bg-main/90"
          >
            {t('continue_to_payment')}
          </Button>
          
          <Button
            type="default"
            size="large"
            block
            onClick={onClose}
            disabled={loading}
          >
            {t('modify_reservation')}
          </Button>
        </Space>
      </div>
    </Modal>
  );
};

export default PriceConfirmationModal;
