'use client';

import { getLimousineLocationErrorMessage, validateLimousineLocation } from "@/utils/limousineLocationValidation";
import {
    GoogleMap,
    Marker,
    StandaloneSearchBox,
    useJsApiLoader,
} from "@react-google-maps/api";
import { message } from "antd";
import { useTranslations } from "next-intl";
import { useEffect, useRef, useState } from "react";
import { CustomInput } from ".";

export default function MapLocationSelector({
  form,
  addressName = "address",
  latName = "latitude",
  lngName = "longitude",
  label,
  placeholder = "ادخل العنوان",
  className,
  onLocationChange,
  address = true,
  watchField = null, // اسم الحقل المراقب للتغييرات
  skipApiValidation = true, // للخدمات الليموزين، نتخطى API validation
  validateOnSelect = true // إذا كان true، نتحقق من الموقع عند التحديد
}) {
  const t = useTranslations();

  // إحداثيات المواقع المختلفة
  const locationCoordinates = {
    'JEDDAH_AIRPORT': { lat: 21.6796, lng: 39.1565, name: 'مطار الملك عبدالعزيز ', cityId: 'JEDDAH' },
    'JEDDAH': { lat: 21.4858, lng: 39.1925, name: 'جدة', cityId: 'JEDDAH' },
    'MAKKAH': { lat: 21.3891, lng: 39.8579, name: 'مكة المكرمة', cityId: 'MAKKAH' }
  };

  const [markerPosition, setMarkerPosition] = useState({
    lat: 21.4858,
    lng: 39.1925,
  });
  const [selectedAddress, setSelectedAddress] = useState("");
  const [manualAddress, setManualAddress] = useState("");
  const [currentCityId, setCurrentCityId] = useState('JEDDAH'); // المدينة الحالية المحددة
  const [locationValidation, setLocationValidation] = useState(null);
  const searchBoxRef = useRef(null);

  const selectedLocation = manualAddress || selectedAddress;

  // مراقبة تغييرات النموذج وتحديث الحالة
  useEffect(() => {
    const existingAddress = form.getFieldValue(addressName);
    const existingLat = form.getFieldValue(latName);
    const existingLng = form.getFieldValue(lngName);

    if (existingAddress) {
      setSelectedAddress(existingAddress);
    }

    if (existingLat && existingLng) {
      setMarkerPosition({
        lat: existingLat,
        lng: existingLng
      });
    }
  }, [form, addressName, latName, lngName]);

  // مراقبة تغييرات الحقل المحدد (مثل اختيار المطار)
  useEffect(() => {
    if (watchField) {
      const watchedValue = form.getFieldValue(watchField);
      if (watchedValue && locationCoordinates[watchedValue]) {
        const coords = locationCoordinates[watchedValue];
        setMarkerPosition({
          lat: coords.lat,
          lng: coords.lng
        });
        setSelectedAddress(coords.name);
        // تحديث المدينة الحالية
        if (coords.cityId) {
          setCurrentCityId(coords.cityId);
        }
        form.setFieldsValue({
          [addressName]: coords.name,
          [latName]: coords.lat,
          [lngName]: coords.lng
        });
      }
    }
  }, [form, watchField, addressName, latName, lngName]);

  // دالة لتحويل العنوان إلى العربية
  const convertToArabic = (address) => {
    if (!address) return '';

    const translations = {
      'Jeddah': 'جدة',
      'Mecca': 'مكة',
      'Riyadh': 'الرياض',
      'Makkah': 'مكة المكرمة',
      'Al Balad': 'البلد',
      'District': 'حي',
      'Street': 'شارع',
      'Road': 'طريق',
      'Avenue': 'جادة',
      'Saudi Arabia': 'المملكة العربية السعودية',
      'Kingdom': 'المملكة',
      'Arabia': 'العربية السعودية'
    };

    let arabicAddress = address;
    Object.entries(translations).forEach(([english, arabic]) => {
      const regex = new RegExp(english, 'gi');
      arabicAddress = arabicAddress.replace(regex, arabic);
    });

    return arabicAddress;
  };

  const { isLoaded } = useJsApiLoader({
    googleMapsApiKey: "AIzaSyAtUOb461InzoQoGEVKKVqqLf2NbwSjqdk",
    libraries: ["places"],
  });

  const reverseGeocode = async (lat, lng, skipValidation = false) => {
    try {
      const res = await fetch(
        `https://maps.googleapis.com/maps/api/geocode/json?latlng=${lat},${lng}&key=AIzaSyAtUOb461InzoQoGEVKKVqqLf2NbwSjqdk&language=ar`
      );
      const data = await res.json();
      if (data.status === "OK") {
        const address = data.results[0]?.formatted_address;
        const arabicAddress = convertToArabic(address || "");
        setSelectedAddress(arabicAddress);
        
        // إذا كان skipApiValidation مفعل و validateOnSelect مفعل، نتحقق من الموقع محليًا
        if (skipApiValidation && validateOnSelect && !skipValidation) {
          const validationResult = validateLimousineLocation(lat, lng, currentCityId);
          setLocationValidation(validationResult);
          
          if (!validationResult.isValid) {
            const errorMessage = getLimousineLocationErrorMessage(validationResult, 'ar');
            message.warning(errorMessage);
            // لا نحفظ الموقع إذا كان غير صالح
            return;
          }
        }
        
        form.setFieldsValue({
          [addressName]: arabicAddress,
          [latName]: lat,
          [lngName]: lng
        });
        if (onLocationChange) {
          onLocationChange({ address: arabicAddress, lat, lng });
        }
      } else {
        setSelectedAddress("");
      }
    } catch (err) {
      // Error fetching address
    }
  };

  const handleMapClick = (e) => {
    const lat = e.latLng?.lat();
    const lng = e.latLng?.lng();
    if (lat && lng) {
      setMarkerPosition({ lat, lng });
      reverseGeocode(lat, lng);
    }
  };

  const handleSearchBoxChange = () => {
    const places = searchBoxRef.current.getPlaces();
    if (places.length > 0) {
      const location = places[0].geometry?.location;
      if (location) {
        const lat = location.lat();
        const lng = location.lng();
        setMarkerPosition({ lat, lng });
        reverseGeocode(lat, lng);
      }
    }
  };

  const handleAddressChange = (e) => {
    const value = e.target.value;
    setManualAddress(value);
    form.setFieldsValue({ [addressName]: value });
  };

  if (!isLoaded) {
    return (
      <div className="w-full h-[300px] bg-gray-200 flex items-center justify-center rounded">
        <span className="text-gray-500">جارٍ تحميل الخريطة...</span>
      </div>
    );
  }

  return (
    <div className="w-full">
      {/* Input Field */}


      {/* Map Container */}
      <div className={`relative w-full h-[300px] rounded   overflow-hidden border border-gray-300 ${className}`}>
        <StandaloneSearchBox
          onLoad={(ref) => (searchBoxRef.current = ref)}
          onPlacesChanged={handleSearchBoxChange}
        >
          <input
            type="text"
            placeholder="ابحث عن الموقع"
            className="absolute z-[1] top-3 right-3 w-72 p-2 rounded shadow bg-white border focus:outline-none"
            dir="rtl"
          />
        </StandaloneSearchBox>

        <GoogleMap
          center={markerPosition}
          zoom={15}
          mapContainerStyle={{ width: "100%", height: "100%" }}
          onClick={handleMapClick}
          options={{
            mapTypeControl: false,
            fullscreenControl: false,
            streetViewControl: false,
          }}
        >
          <Marker position={markerPosition} />
        </GoogleMap>

        {/* Select Location Button */}
        <button
          type="button"
          className="absolute bottom-3 left-3 bg-white px-4 py-2 rounded shadow border hover:bg-gray-50 transition-colors"
          onClick={() => {
            // التحقق من الموقع قبل الحفظ
            if (skipApiValidation && validateOnSelect) {
              const validationResult = validateLimousineLocation(
                markerPosition.lat,
                markerPosition.lng,
                currentCityId
              );
              
              if (!validationResult.isValid) {
                const errorMessage = getLimousineLocationErrorMessage(validationResult, 'ar');
                message.error(errorMessage);
                return;
              }
              
              message.success('تم تحديد الموقع بنجاح');
            }
            
            form.setFieldsValue({
              [addressName]: selectedLocation,
              [latName]: markerPosition.lat,
              [lngName]: markerPosition.lng
            });
            if (onLocationChange) {
              onLocationChange({
                address: selectedLocation,
                lat: markerPosition.lat,
                lng: markerPosition.lng
              });
            }
          }}
        >
          حدد الموقع
        </button>
      </div>
      {address && (
        <CustomInput
          type="text"
          value={selectedLocation}
          onChange={handleAddressChange}
          placeholder={placeholder}
          className="w-full h-[50px] border border-[#979797] rounded text-sm mt-5"
          dir="rtl"
        />
      )}
    </div>
  );
}
