// components/AdvancedRentalPanel/inputs/StartTimePicker.jsx
'use client';

import { selectSelectedRange } from '@/store/features/rental/rentalSelectors';
import { ClockCircleOutlined } from '@ant-design/icons';
import { setSavedStartTime, setStartTime } from '@store/features/rental/rentalSlice';
import { Select } from 'antd';
import dayjs from 'dayjs';
import 'dayjs/locale/ar';
import { useLocale, useTranslations } from 'next-intl';
import { useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
// Configure dayjs with Arabic locale
dayjs.locale('ar');

const StartTimePicker = ({ disabled, workingHours = null, selectedBranchId = null, selectedDate = null, onChange: customOnChange, ...props }) => {
  const dispatch = useDispatch();
  const selectedRangeFromStore = useSelector(selectSelectedRange);
  const { startTime } = useSelector((state) => state.rental);
  const locale = useLocale();
  const t = useTranslations();

  // دالة لتحديد اليوم من التاريخ
  const getDayName = (date) => {
    if (!date) return null;
    try {
      const dayjsDate = dayjs(date);
      if (!dayjsDate.isValid()) return null;
      const dayNames = ['sun', 'mon', 'tues', 'wed', 'thurs', 'fri', 'sat'];
      const dayIndex = dayjsDate.day();
      return dayNames[dayIndex];
    } catch (error) {
      return null;
    }
  };

  // دالة لتحويل الوقت إلى صيغة 12 ساعة مع AM/PM
  const formatTime12Hour = (hour, minute) => {
    const period = hour >= 12 ? (locale === 'en' ? 'PM' : 'م') : (locale === 'en' ? 'AM' : 'ص'); // م = مساء، ص = صباح
    const displayHour = hour === 0 ? 12 : hour > 12 ? hour - 12 : hour;
    const displayMinute = minute.toString().padStart(2, '0');
    return {
      time24: `${hour.toString().padStart(2, '0')}:${displayMinute}`,
      time12: `${displayHour}:${displayMinute} ${period}`,
      display: `${displayHour}:${displayMinute} ${period}`
    };
  };

  // دالة لتوليد الأوقات المتاحة
  const getAvailableTimes = useMemo(() => {
    const times = [];

    // تحديد التاريخ للاستخدام - إما من selectedDate أو من selectedRangeFromStore
    let dateToUse = selectedDate;
    if (!dateToUse && selectedRangeFromStore && selectedRangeFromStore.length > 0) {
      dateToUse = selectedRangeFromStore[0];
    }

    // إذا لم تتوفر البيانات، إرجاع أوقات افتراضية مع validation للوقت الحالي
    if (!workingHours || !selectedBranchId || !dateToUse) {
      const now = dayjs();
      const isToday = dateToUse && dayjs(dateToUse).isSame(now, 'day');

      for (let hour = 0; hour < 24; hour++) {
        for (let min = 0; min < 60; min += 30) {
          const timeInfo = formatTime12Hour(hour, min);

          let isDisabled = false;
          // إذا كان التاريخ اليوم، تعطيل الأوقات السابقة
          if (isToday) {
            const currentHour = now.hour();
            const currentMinute = now.minute();

            if (hour < currentHour || (hour === currentHour && min <= currentMinute)) {
              isDisabled = true;
            }
          }

          times.push({
            value: timeInfo.time24,
            label: timeInfo.display,
            disabled: isDisabled
          });
        }
      }
      return times;
    }

    const startDate = typeof dateToUse === 'string'
      ? dayjs(dateToUse)
      : dateToUse;

    if (!startDate || !startDate.isValid()) {
      // توليد أوقات افتراضية
      for (let hour = 0; hour < 24; hour++) {
        for (let min = 0; min < 60; min += 30) {
          const timeInfo = formatTime12Hour(hour, min);
          times.push({
            value: timeInfo.time24,
            label: timeInfo.display,
            disabled: false
          });
        }
      }
      return times;
    }

    const dayName = getDayName(startDate);
    if (!dayName) return times;

    const dayKey = dayName.toLowerCase();
    const isOpen = workingHours[dayKey] > 0;

    if (!isOpen) {
      // الفرع مغلق، إرجاع قائمة فارغة أو رسالة
      return [];
    }

    const openHour = workingHours[`${dayKey}OpenHour`];
    const openMin = workingHours[`${dayKey}OpenMin`];
    const closeHour = workingHours[`${dayKey}CloseHour`];
    const closeMin = workingHours[`${dayKey}CloseMin`];

    // التأكد من وجود قيم صحيحة
    if (openHour === undefined || openMin === undefined ||
      closeHour === undefined || closeMin === undefined) {
      return times;
    }

    const now = dayjs();
    const isToday = startDate.isSame(now, 'day');

    // توليد جميع الأوقات الممكنة وتحديد المعطل منها
    for (let hour = 0; hour < 24; hour++) {
      for (let min = 0; min < 60; min += 30) {
        const timeInfo = formatTime12Hour(hour, min);

        let isDisabled = false;

        // تعطيل الأوقات قبل فتح الفرع
        if (hour < openHour || (hour === openHour && min < openMin)) {
          isDisabled = true;
        }

        // تعطيل الأوقات بعد إغلاق الفرع
        if (hour > closeHour || (hour === closeHour && min > closeMin)) {
          isDisabled = true;
        }

        // إذا كان اليوم، تعطيل الأوقات السابقة
        if (isToday) {
          const currentHour = now.hour();
          const currentMinute = now.minute();

          if (hour < currentHour || (hour === currentHour && min <= currentMinute)) {
            isDisabled = true;
          }
        }

        // إضافة الوقت فقط إذا كان ضمن ساعات العمل أو خارجها (لإظهار جميع الخيارات)
        times.push({
          value: timeInfo.time24,
          label: timeInfo.display,
          disabled: isDisabled
        });
      }
    }

    return times;
  }, [workingHours, selectedBranchId, selectedRangeFromStore, selectedDate]);

  // دالة لعرض القيمة المختارة بصيغة 12 ساعة
  const getDisplayValue = (timeValue) => {
    if (!timeValue) return timeValue;
    const [hour, minute] = timeValue.split(':');
    const timeInfo = formatTime12Hour(parseInt(hour), parseInt(minute));
    return timeInfo.display;
  };

  // Label للوصولية
  const ariaLabel = locale === 'en' 
    ? 'Select rental start time' 
    : t('rental_start_time') || 'وقت بدء الإيجار';

  return (
    <Select
      value={customOnChange ? (props.value ? getDisplayValue(props.value) : props.value) : (startTime ? getDisplayValue(startTime) : startTime)}
      onChange={(value) => {
        // إذا كان هناك customOnChange، استخدمه (للليموزين)
        if (customOnChange) {
          customOnChange(value);
        } else {
          // إذا لم يكن هناك customOnChange، استخدم Redux (للتأجير)
          dispatch(setStartTime(value));
          dispatch(setSavedStartTime(value));
        }
      }}
      placeholder={t('rental_start_time')}
      aria-label={ariaLabel}
      size='large'
      disabled={disabled}
      suffixIcon={<ClockCircleOutlined />}
      options={getAvailableTimes}
      showSearch
      optionFilterProp="label"
      filterOption={(input, option) =>
        (option?.label ?? '').toLowerCase().includes(input.toLowerCase())
      }
      {...props}
      className="w-full [&_svg]:!mt-2 h-[50px] max-sm:!h-[40px] [&_.ant-select-selector]:!h-[50px] [&_.ant-select-selector]:!border-[#162A2B] [&_.ant-select-selector]:!rounded max-sm:[&_.ant-select-selector]:!h-[40px] [&_.ant-select-selection-item]:!leading-[50px] max-sm:[&_.ant-select-selection-item]:!leading-[40px] [&_.ant-select-selection-placeholder]:!leading-[50px] max-sm:[&_.ant-select-selection-placeholder]:!leading-[40px] max-sm:[&_.ant-select-selection-item]:!text-xs max-sm:[&_.ant-select-selection-placeholder]:!text-xs"
      classNames={{
        popup: {
          root: "[&_.ant-select-item]:!py-2 max-sm:[&_.ant-select-item]:!text-xs"
        }
      }}
      // popupClassName="[&_.ant-select-item]:!py-2 max-sm:[&_.ant-select-item]:!text-xs"
      notFoundContent={
        <div className="text-center py-4 text-gray-500">
          {t('no_available_times') || 'لا توجد أوقات متاحة'}
        </div>
      }
    />
  );
};

export default StartTimePicker;
