'use client';

import { selectDuration, selectSelectedRange } from '@/store/features/rental/rentalSelectors';
import { setSavedEndDate, setSavedStartDate, setSelectedRange } from '@/store/features/rental/rentalSlice';
import customArabicLocale from '@ui/arabicDateLocale';
import { Button, ConfigProvider, DatePicker } from 'antd';
import dayjs from 'dayjs';
import 'dayjs/locale/ar';
import { useLocale, useTranslations } from 'next-intl';
import { useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';

const { RangePicker } = DatePicker;

dayjs.locale('ar');

const DateRangePicker = ({ workingHours = null, selectedBranchId = null }) => {
  let locale = useLocale()
  const t = useTranslations();

  const dispatch = useDispatch();
  const selectedRangeFromStore = useSelector(selectSelectedRange);
  const duration = useSelector(selectDuration);
  const { settings, membership } = useSelector((state) => state.shared);

  // State للـ error
  const [error, setError] = useState('');

  // Convert ISO strings back to dayjs objects for the component
  const selectedRange = selectedRangeFromStore.map(date =>
    typeof date === 'string' ? dayjs(date) : date
  );

  //❗ منطق الـ validation مشابه لـ FromDateSelector
  const getValidationHours = () => {
    // التحقق من preOrderAllowed واستخدام hoursPreOrder إذا كان متاحاً
    if (membership?.preOrderAllowed == 1 && membership?.hoursPreOrder) {
      return membership.hoursPreOrder;
    } else if (settings?.NUM_HOURS_FOR_START_RESERVATION_FROM_BRANCH) {
      return settings.NUM_HOURS_FOR_START_RESERVATION_FROM_BRANCH;
    }
    return null;
  };

  const validationHours = getValidationHours();

  // دالة لتحديد اليوم من التاريخ
  const getDayName = (date) => {
    if (!date) return null;
    try {
      const dayjsDate = dayjs(date);
      if (!dayjsDate.isValid()) return null;
      const dayNames = ['sun', 'mon', 'tues', 'wed', 'thurs', 'fri', 'sat'];
      const dayIndex = dayjsDate.day();
      return dayNames[dayIndex];
    } catch (error) {
      return null;
    }
  };

  // دالة للتحقق من أن الفرع مفتوح في يوم معين
  const isBranchOpenOnDay = (date) => {
    if (!workingHours || !selectedBranchId || !date) return true; // افتراضي مفتوح إذا لم تتوفر البيانات
    
    const dayName = getDayName(date);
    if (!dayName) return true;
    
    const dayKey = dayName.toLowerCase();
    const isOpen = workingHours[dayKey] > 0; 
    
    return isOpen; // إذا كان الفرع مفتوح في هذا اليوم
  };

  const disabledDate = (current) => {
    if (!current) return false;

    const now = dayjs();
    const today = now.startOf('day');

    // تعطيل التواريخ السابقة
    if (current < today) return true;

    // تعطيل الأيام التي يكون فيها الفرع مغلق
    if (!isBranchOpenOnDay(current)) return true;

    return false;
  };

  // دالة لتحويل الرقم إلى نص عربي مناسب
  const getDaysText = (days) => {
    if (days === 1) {
      return t('date_picker.one_day');
    } else if (days === 2) {
      return t('date_picker.two_days');
    } else {
      return t('date_picker.days', { days });
    }
  };

  // دالة للتحقق من صحة التواريخ المختارة
  const validateDates = (dates) => {
    if (!dates || dates.length === 0) {
      setError('');
      return true;
    }

    const now = dayjs();
    const startDate = dates[0];
    const endDate = dates[1];

    // التحقق من start date
    if (validationHours && startDate) {
      // إضافة يوم إضافي إلى المدة المسموحة
      const maxAllowedDateTime = now.add(validationHours, 'hour').add(0, 'day');

      if (startDate.isAfter(maxAllowedDateTime, 'day')) {
        const allowedDays = Math.ceil(validationHours / 24) ;
        setError(t('date_picker.cannot_start_after', { days: getDaysText(allowedDays) }));
        // مسح التاريخ من Redux عند وجود خطأ
        dispatch(setSelectedRange([]));
        dispatch(setSavedStartDate(null));
        dispatch(setSavedEndDate(null));
        return false;
      }
    }

    // التحقق من الحد الأدنى لعدد أيام الحجز
    if (startDate && endDate && settings?.MIN_DAYS_RESERVATION) {
      const selectedDays = endDate.diff(startDate, 'day') + 1;
      const minDays = parseInt(settings.MIN_DAYS_RESERVATION);
      
      if (selectedDays <= minDays) {
        setError(t('validation.min_days_reservation', { 
          minDays: getDaysText(minDays), 
          selectedDays: getDaysText(selectedDays) 
        }));
        // مسح التاريخ من Redux عند وجود خطأ
        dispatch(setSelectedRange([]));
        dispatch(setSavedStartDate(null));
        dispatch(setSavedEndDate(null));
        return false;
      }
    }

    setError('');
    return true;
  };

  const setRange = (days) => {
    const start = selectedRange[0] || dayjs();
    const end = start.add(days - 1, 'day');

    // التحقق من الحد الأدنى قبل التحقق من صحة النطاق
    if (settings?.MIN_DAYS_RESERVATION) {
      const minDays = parseInt(settings.MIN_DAYS_RESERVATION);
      if (days < minDays) {
        setError(t('validation.min_days_reservation', { 
          minDays: getDaysText(minDays), 
          selectedDays: getDaysText(days) 
        }));
        return;
      }
    }

    // التحقق من صحة النطاق
    const dates = [start, end];
    const isValid = validateDates(dates);

    if (isValid) {
      // Convert to date strings without timezone issues
      const serializedRange = [start.format('YYYY-MM-DD'), end.format('YYYY-MM-DD')];
      dispatch(setSelectedRange(serializedRange));
      
      // حفظ التواريخ فوراً للاستخدام في صفحة الحجز
      dispatch(setSavedStartDate(serializedRange[0]));
      dispatch(setSavedEndDate(serializedRange[1]));
    }
  };

  const renderFooter = () => {
    const btns = {
      daily: [
        { label: t('date_picker.quick_actions.plus_day'), days: 2 },
        { label: t('date_picker.quick_actions.plus_two_days'), days: 3 },
        { label: t('date_picker.quick_actions.plus_3_days'), days: 4 },
        { label: t('date_picker.quick_actions.plus_5_days'), days: 6 },
      ],
      weekly: [
        { label: t('date_picker.quick_actions.plus_week'), days: 8 },
        { label: t('date_picker.quick_actions.plus_two_weeks'), days: 15 },
        { label: t('date_picker.quick_actions.plus_3_weeks'), days: 22 },
      ],
      monthly: [
        { label: t('date_picker.quick_actions.plus_month'), days: 30 },
        { label: t('date_picker.quick_actions.plus_3_months'), days: 90 },
        { label: t('date_picker.quick_actions.plus_6_months'), days: 180 },
        { label: t('date_picker.quick_actions.plus_9_months'), days: 270 },
      ],
    };

    return (
      <div className="flex gap-2 flex-col justify-center py-4">
          {error && (
            <div className="mt-2 text-red-500 text-center text-sm font-medium mb-2">
              {error}
            </div>
          )}
        <div className="flex flex-wrap justify-center items-center gap-2">
          {(btns[duration] || []).map((btn, idx) => (
            <Button
              key={idx}
              size="small"
              className="bg-main text-white px-3 py-[14px]"
              onClick={() => setRange(btn.days)}
            >
              {btn.label}
            </Button>
          ))}
        </div>
      </div>
    );
  };

  return (
    <div className="w-full">
      <ConfigProvider direction="rtl">
        <RangePicker
          locale={locale == "ar" && customArabicLocale}
          disabledDate={disabledDate}
          format="DD/MM/YYYY"
          placeholder={[t('date_picker.start_placeholder'), t('date_picker.end_placeholder')]}
          size="large"
          value={selectedRange}
          onChange={(dates) => {
            // التحقق من صحة التواريخ
            const isValid = validateDates(dates);

            if (dates && dates.length > 0 && isValid) {
              // Convert dayjs objects to date strings without timezone issues
              const serializedDates = dates.map(date => date.format('YYYY-MM-DD'));
              dispatch(setSelectedRange(serializedDates));
              
              // حفظ التواريخ فوراً للاستخدام في صفحة الحجز
              dispatch(setSavedStartDate(serializedDates[0]));
              dispatch(setSavedEndDate(serializedDates[1]));
            } else if (!dates || dates.length === 0) {
              dispatch(setSelectedRange([]));
              dispatch(setSavedStartDate(null));
              dispatch(setSavedEndDate(null));
              setError('');
            }
            // إذا كان غير صحيح، لا نحفظ في Redux ونعرض الخطأ
          }}
          renderExtraFooter={renderFooter}
          // popupClassName="[&_.ant-picker-panels]:max-md:flex-col"
          classNames={{
            popup: {
              root: "[&_.ant-picker-panels]:max-md:flex-col",
            },
          }}
          className={`w-full h-[50px] [&_.ant-picker-large .ant-picker-input>input]:max-sm:!text-xs rounded text-sm max-sm:!h-[40px] max-sm:!text-xs border ${error ? 'border-red-500' : 'border-[#162A2B]'}`}
        />
      </ConfigProvider>

      {/* عرض التاريخ المختار */}
      {selectedRangeFromStore && selectedRangeFromStore.length === 2 && selectedRangeFromStore[0] && selectedRangeFromStore[1] && (
        <div className="mt-2 text-xs text-gray-600 ">
          <span className="font-medium">{t('date_picker.selected_date')}</span>
          <span className="text-main font-semibold">
            {t('date_picker.from')} {dayjs(selectedRangeFromStore[0]).format('DD/MM/YYYY')} {t('date_picker.to')} {dayjs(selectedRangeFromStore[1]).format('DD/MM/YYYY')}
          </span>
          <span className="text-gray-500">
            {' '}({dayjs(selectedRangeFromStore[1]).diff(dayjs(selectedRangeFromStore[0]), 'day')  } {t('date_picker.day_unit')})
          </span>
        </div>
      )}

      {/* عرض رسالة الخطأ */}
      {error && (
        <div className="mt-2 text-xs text-red-500 font-medium  ">
          {error}
        </div>
      )}

    </div>
  );
};

export default DateRangePicker;
