'use client';

import useLocationSelection from '@/hook/useLocationSelection';
import { setPickupLocation, setSelectedCarTypes, setSelectedCityId, setSelectedRange, setStartTime } from '@/store/features/rental/rentalSlice';
import { CaretRightOutlined, SearchOutlined } from '@ant-design/icons';
import StartTimePicker from '@components/AdvancedRentalPanel/inputs/StartTimePicker';
import DeliveryLocation from '@components/ui/googlemap/Location';
import { Button, Checkbox, Col, Form, Modal, Row, message } from 'antd';
import dayjs from 'dayjs';
import 'dayjs/locale/ar';
import { useLocale, useTranslations } from 'next-intl';
import { useEffect, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import CitySelector from '../../inputs/CitySelector';
import DateRangePicker from '../../inputs/DateRangePicker';
import CarTypeSelector from './bloc/CarTypeSelector';

dayjs.locale('ar');

const RentalDeliveryForm = () => {
  const t = useTranslations();
  const locale = useLocale();
  const [form] = Form.useForm();
  const [open, setOpen] = useState(false);
  const [locationSet, setLocationSet] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const closeModal = () => setOpen(false);
  const { carTypes } = useSelector((state) => state.shared);
  const { selectedCarTypes, selectedRange, startTime, pickupAddress, pickupLatitude, pickupLongitude } = useSelector((state) => state.rental);
  const dispatch = useDispatch();
  const {
    selectedCityId,
    handleCityChange,
    handleDataChange
  } = useLocationSelection((locationData) => {
    // حفظ بيانات الموقع من الخريطة
    const formLocationData = form.getFieldsValue(['pickupAddress', 'pickupLatitude', 'pickupLongitude']);
  });

  // استرجاع البيانات المحفوظة من sessionStorage عند التحميل
  useEffect(() => {
    if (typeof window !== 'undefined') {
      try {
        const savedData = sessionStorage.getItem('rental_delivery_form');
        if (savedData) {
          const parsedData = JSON.parse(savedData);
          
          // استرجاع المدينة
          if (parsedData.selectedCityId) {
            dispatch(setSelectedCityId(parsedData.selectedCityId));
          }
          
          // استرجاع التواريخ - تحويلها إلى صيغة {YYYY-MM-DD} للتوافق مع DateRangePicker
          if (parsedData.selectedRange && Array.isArray(parsedData.selectedRange)) {
            const range = parsedData.selectedRange
              .filter(date => date !== null && date !== undefined)
              .map(date => {
                try {
                  // تحويل التاريخ إلى dayjs object ثم إلى صيغة {YYYY-MM-DD}
                  const dayjsDate = dayjs(date);
                  return dayjsDate.isValid() ? dayjsDate.format('YYYY-MM-DD') : null;
                } catch (error) {
                  console.error('Error parsing date:', error);
                  return null;
                }
              })
              .filter(date => date !== null);
            
            if (range.length === 2) {
              dispatch(setSelectedRange(range));
            }
          }
          
          // استرجاع الوقت
          if (parsedData.startTime) {
            dispatch(setStartTime(parsedData.startTime));
          }
          
          // استرجاع أنواع السيارات المختارة
          if (parsedData.selectedCarTypes && Array.isArray(parsedData.selectedCarTypes)) {
            dispatch(setSelectedCarTypes(parsedData.selectedCarTypes));
          }
          
          // استرجاع موقع الاستلام
          if (parsedData.pickupAddress || parsedData.pickupLatitude || parsedData.pickupLongitude) {
            dispatch(setPickupLocation({
              address: parsedData.pickupAddress,
              latitude: parsedData.pickupLatitude,
              longitude: parsedData.pickupLongitude
            }));
            
            // تعيين قيم النموذج
            form.setFieldsValue({
              pickupAddress: parsedData.pickupAddress,
              pickupLatitude: parsedData.pickupLatitude,
              pickupLongitude: parsedData.pickupLongitude
            });
            
            if (parsedData.pickupAddress && parsedData.pickupLatitude && parsedData.pickupLongitude) {
              setLocationSet(true);
            }
          }
        }
      } catch (error) {
        console.error('Error loading rental delivery form data:', error);
      }
    }
  }, [dispatch, form]);

  // حفظ البيانات في sessionStorage عند التغيير
  useEffect(() => {
    if (typeof window !== 'undefined') {
      try {
        // التأكد من وجود بيانات قبل الحفظ
        if (selectedCityId || (selectedRange && selectedRange.length > 0) || startTime) {
          const formValues = form.getFieldsValue(['pickupAddress', 'pickupLatitude', 'pickupLongitude']);
          const dataToSave = {
            selectedCityId,
            // حفظ التواريخ كما هي (strings بصيغة {YYYY-MM-DD} أو ISO)
            selectedRange: selectedRange && selectedRange.length > 0 
              ? selectedRange.map(date => {
                  if (!date) return null;
                  // إذا كان التاريخ string، احفظه مباشرة
                  if (typeof date === 'string') return date;
                  // إذا كان dayjs object، حوله إلى string
                  if (date && typeof date === 'object' && date.format) {
                    return date.format('YYYY-MM-DD');
                  }
                  return null;
                })
              : null,
            startTime,
            selectedCarTypes,
            pickupAddress: formValues.pickupAddress,
            pickupLatitude: formValues.pickupLatitude,
            pickupLongitude: formValues.pickupLongitude,
            timestamp: new Date().toISOString()
          };
          sessionStorage.setItem('rental_delivery_form', JSON.stringify(dataToSave));
        }
      } catch (error) {
        console.error('Error saving rental delivery form data:', error);
      }
    }
  }, [selectedCityId, selectedRange, startTime, selectedCarTypes, form]);


  // التحقق من اكتمال جميع البيانات المطلوبة
  const isFormComplete = () => {
    // التحقق من قيم النموذج مباشرة
    const formValues = form.getFieldsValue(['pickupAddress', 'pickupLatitude', 'pickupLongitude']);
    const hasLocation = formValues.pickupAddress && formValues.pickupLatitude && formValues.pickupLongitude;
    
    return selectedCityId &&
      selectedRange &&
      selectedRange.length === 2 &&
      selectedRange[0] &&
      selectedRange[1] &&
      startTime &&
      hasLocation;
  };

  const showModal = () => {
    if (!isFormComplete()) {
      message.warning(t('fill_all_required_data'));
      return;
    }
    setOpen(true);
  };

  const handleSelectionChange = (typeId, isSelected) => {
    const currentSelectedCarTypes = selectedCarTypes || [];

    if (isSelected) {
      const newSelectedCarTypes = [...currentSelectedCarTypes, typeId];
      dispatch(setSelectedCarTypes(newSelectedCarTypes));
    } else {
      const newSelectedCarTypes = currentSelectedCarTypes.filter(id => id !== typeId);
      dispatch(setSelectedCarTypes(newSelectedCarTypes));
    }
  };

  const handleSearch = async () => {
    try {
      setIsLoading(true);
      
      // الحصول على بيانات الموقع من النموذج
      const locationData = form.getFieldsValue(['pickupAddress', 'pickupLatitude', 'pickupLongitude']);

      // التحقق من أن الموقع تم اختياره
      if (!locationData.pickupAddress || !locationData.pickupLatitude || !locationData.pickupLongitude) {
        message.warning(t('select_delivery_location_first'));
        setIsLoading(false);
        return;
      }

      if (selectedCarTypes && selectedCarTypes.length > 0) {
        // تمرير بيانات الموقع إلى handleDataChange
        await handleDataChange(locationData);
        // إبقاء الـ loading نشط حتى يتم الانتقال للصفحة التالية
        // سيتم إيقافه تلقائياً عند إعادة تحميل الصفحة أو الانتقال
      } else {
        // يمكنك إضافة رسالة تنبيه للمستخدم هنا
        setIsLoading(false);
      }
    } catch (error) {
      // Handle error silently
      setIsLoading(false);
    }
  };

  const handleSelectAll = () => {
    try {
      if (carTypes && carTypes.length > 0) {
        const allCarTypeIds = carTypes.map(carType => carType.typeId || carType.id);
        dispatch(setSelectedCarTypes(allCarTypeIds));
      }
    } catch (error) {
      // Handle error silently
    }
  };

  const handleDeselectAll = () => {
    try {
      dispatch(setSelectedCarTypes([])); 
    } catch (error) {
      // Handle error silently
    }
  };

  const isAllSelected = selectedCarTypes && carTypes && selectedCarTypes.length === carTypes.length;
  const hasAnySelected = selectedCarTypes && selectedCarTypes.length > 0;

  return (
    <>
    <div className="pt-4">
      <Form
        form={form}
        onValuesChange={(changedValues, allValues) => {
          // تحديث حالة الموقع عندما يتغير
          if (changedValues.pickupAddress || changedValues.pickupLatitude || changedValues.pickupLongitude) {
            const hasLocation = allValues.pickupAddress && allValues.pickupLatitude && allValues.pickupLongitude;
            setLocationSet(hasLocation);
            
            // حفظ موقع الاستلام في Redux
            dispatch(setPickupLocation({
              address: allValues.pickupAddress,
              latitude: allValues.pickupLatitude,
              longitude: allValues.pickupLongitude
            }));
            
            // حفظ البيانات في sessionStorage
            if (typeof window !== 'undefined') {
              try {
                const dataToSave = {
                  selectedCityId,
                  // حفظ التواريخ كما هي (strings بصيغة {YYYY-MM-DD} أو ISO)
                  selectedRange: selectedRange && selectedRange.length > 0 
                    ? selectedRange.map(date => {
                        if (!date) return null;
                        // إذا كان التاريخ string، احفظه مباشرة
                        if (typeof date === 'string') return date;
                        // إذا كان dayjs object، حوله إلى string
                        if (date && typeof date === 'object' && date.format) {
                          return date.format('YYYY-MM-DD');
                        }
                        return null;
                      })
                    : null,
                  startTime,
                  selectedCarTypes,
                  pickupAddress: allValues.pickupAddress,
                  pickupLatitude: allValues.pickupLatitude,
                  pickupLongitude: allValues.pickupLongitude,
                  timestamp: new Date().toISOString()
                };
                sessionStorage.setItem('rental_delivery_form', JSON.stringify(dataToSave));
              } catch (error) {
                console.error('Error saving form data:', error);
              }
            }
          }
        }}
      >
        <Row gutter={[20, 20]} align="top">
          <Col xl={4} lg={4} md={6} sm={12} xs={12}>
            <CitySelector
              value={selectedCityId}
              onChange={handleCityChange}
              placeholder={t('select_city')}
              className="w-full h-[50px] border !border-[#162A2B] rounded text-sm max-sm:!h-[40px] [&_.ant-select-selector]:max-sm:!text-xs"
            />
          </Col>

          <Col xl={6} lg={6} md={12} sm={12} xs={12} className='DeliveryLocation' >
            <DeliveryLocation
              form={form}
              addressName="pickupAddress"
              latName="pickupLatitude"
              lngName="pickupLongitude"
              label={t('delivery_location_label')}
              className="text-xs truncate  max-sm:!text-xs h-[50px] border border-[#162A2B] rounded-[3.5px] max-sm:!h-[40px] m-0"
            />
          </Col>
          <Col xl={7} lg={5} md={6} sm={24} xs={24}>
            <DateRangePicker
              locale={locale}
            />
          </Col>

          <Col xl={4} lg={4} md={6} sm={12} xs={12}>
            <StartTimePicker />
          </Col>


          <Col xl={3} lg={3} md={6} sm={12} xs={12}>
            <Button
              size="large"
              type="primary"
              className="!bg-[#162A2B] h-[50px]  !text-white px-4 md:px-10 w-full text-sm md:text-base disabled:!bg-gray-400 disabled:cursor-not-allowed max-sm:!h-[40px]   max-sm:!text-xs max-sm:!px-2 max-sm:!font-medium"
              onClick={showModal}
              // disabled={!isFormComplete()}
            >
              {t('next')}

            </Button>
          </Col>
        </Row>
      </Form>

      <Modal
        title={<span className="text-lg font-semibold text-center w-full block">{t('select_car_type_title')}</span>}
        open={open}
        onCancel={closeModal}
        footer={
          <div className="flex justify-center items-center gap-3 pt-2">
            <Button
              type="primary"
              className='!bg-[#fff] !text-main shadow-none border border-main h-[40px] w-full md:w-[100px] text-sm md:text-base max-sm:!h-6 max-sm:!text-xs max-sm:!font-medium'
              icon={<CaretRightOutlined />}
              onClick={async () => {
                try {
                  setIsLoading(true);
                  
                  // إلغاء أي اختيارات سابقة للأنواع
                  dispatch(setSelectedCarTypes([]));
                  
                  // الحصول على بيانات الموقع من النموذج
                  const locationData = form.getFieldsValue(['pickupAddress', 'pickupLatitude', 'pickupLongitude']);
                  
                  // تخطي بدون اختيار أي أنواع سيارات - تمرير مصفوفة فارغة مباشرة
                  await handleDataChange(locationData, []);
                } catch (error) {
                  // Handle error silently
                  setIsLoading(false);
                }
              }}>
              {t('skip')}
            </Button>
            <Button
              type="primary"
              className='shadow-none h-[40px] w-full md:w-[250px] text-sm md:text-base max-sm:!h-6 max-sm:!text-xs max-sm:!font-medium'
              icon={<SearchOutlined />}
              onClick={handleSearch}
              loading={isLoading}
              disabled={isLoading}
            >
              {isLoading ? t('searching') : t('search_for_results')}
            </Button>
          </div>
        }
        centered
      >
        <p className='text-center text-sm text-gray-600 mb-4'>{t('select_car_type')}</p>

        {/* Select All / Deselect All Checkboxes */}
        {carTypes && carTypes.length > 0 && (
          <div className="mt-4 p-2 border border-gray-200 rounded-lg bg-gray-50/50">
            <div className="flex items-center justify-between">
              <Checkbox
                checked={isAllSelected}
                onChange={handleSelectAll}
                className="!text-primary"
              >
                <span className="text-sm text-gray-700 font-medium">
                  {isAllSelected ? t('deselect_all') : t('select_all')}
                </span>
              </Checkbox>
              
              <span className="text-xs text-gray-500 bg-white px-2 py-1 rounded-full">
                {selectedCarTypes?.length || 0}/{carTypes.length}
              </span>
            </div>
          </div>
        )}

        <Row gutter={[12, 12]} className='mt-8 '>
          {carTypes && carTypes.length > 0 ? (
            carTypes.map((carType) => (
              <Col lg={12} md={12} sm={12} xs={12} key={carType.typeId || carType.id}>
                <CarTypeSelector
                  locale={locale}
                  data={carType}
                  onSelectionChange={(typeId, isSelected) => handleSelectionChange(typeId, isSelected)}
                />
              </Col>
            ))
          ) : (
            <Col span={24}>
              <div className="text-center text-gray-500 py-8 text-sm md:text-base max-sm:!text-xs max-sm:!font-medium">
                {t('no_car_types_available') || 'لا توجد أنواع سيارات متاحة'}
              </div>
            </Col>
          )}
        </Row>
      </Modal>
    </div>
    </>
  );  
};

export default RentalDeliveryForm;
