'use client';

import useBranchWorkingHours from '@/hook/useBranchWorkingHours';
import useLocationSelection from '@/hook/useLocationSelection';
import { setSelectedBranchId, setSelectedCarTypes, setSelectedCityId, setSelectedRange, setStartTime } from '@/store/features/rental/rentalSlice';
import { CaretRightOutlined, SearchOutlined } from '@ant-design/icons';
import { Button, Checkbox, Col, Modal, Row } from 'antd';
import dayjs from 'dayjs';
import 'dayjs/locale/ar';
import { useLocale, useTranslations } from 'next-intl';
import { useEffect, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import BranchSelector from '../../inputs/BranchSelector';
import CitySelector from '../../inputs/CitySelector';
import DateRangePicker from '../../inputs/DateRangePicker';
import StartTimePicker from '../../inputs/StartTimePicker';
import CarTypeSelector from './bloc/CarTypeSelector';

dayjs.locale('ar');

const RentalBranchForm = () => {
  const t = useTranslations();
  const locale = useLocale();
  const [open, setOpen] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const closeModal = () => setOpen(false);
  const { carTypes } = useSelector((state) => state.shared)
  const { selectedCarTypes, selectedRange, startTime } = useSelector((state) => state.rental)
  const dispatch = useDispatch();
  const {
    selectedCityId,
    selectedBranchId,
    handleCityChange,
    handleBranchChange,
    handleDataChange
  } = useLocationSelection();

  // استرجاع البيانات المحفوظة من sessionStorage عند التحميل
  useEffect(() => {
    if (typeof window !== 'undefined') {
      try {
        const savedData = sessionStorage.getItem('rental_branch_form');
        if (savedData) {
          const parsedData = JSON.parse(savedData);
          
          // استرجاع المدينة والفرع
          if (parsedData.selectedCityId) {
            dispatch(setSelectedCityId(parsedData.selectedCityId));
          }
          if (parsedData.selectedBranchId) {
            dispatch(setSelectedBranchId(parsedData.selectedBranchId));
          }
          
          // استرجاع التواريخ - تحويلها إلى صيغة 'YYYY-MM-DD' للتوافق مع DateRangePicker
          if (parsedData.selectedRange && Array.isArray(parsedData.selectedRange)) {
            const range = parsedData.selectedRange
              .filter(date => date !== null && date !== undefined)
              .map(date => {
                try {
                  // تحويل التاريخ إلى dayjs object ثم إلى صيغة 'YYYY-MM-DD'
                  const dayjsDate = dayjs(date);
                  return dayjsDate.isValid() ? dayjsDate.format('YYYY-MM-DD') : null;
                } catch (error) {
                  console.error('Error parsing date:', error);
                  return null;
                }
              })
              .filter(date => date !== null);
            
            if (range.length === 2) {
              dispatch(setSelectedRange(range));
            }
          }
          
          // استرجاع الوقت
          if (parsedData.startTime) {
            dispatch(setStartTime(parsedData.startTime));
          }
          
          // استرجاع أنواع السيارات المختارة
          if (parsedData.selectedCarTypes && Array.isArray(parsedData.selectedCarTypes)) {
            dispatch(setSelectedCarTypes(parsedData.selectedCarTypes));
          }
        }
      } catch (error) {
        console.error('Error loading rental branch form data:', error);
      }
    }
  }, [dispatch]);

  // حفظ البيانات في sessionStorage عند التغيير
  useEffect(() => {
    if (typeof window !== 'undefined') {
      try {
        // التأكد من وجود بيانات قبل الحفظ
        if (selectedCityId || selectedBranchId || (selectedRange && selectedRange.length > 0) || startTime) {
          const dataToSave = {
            selectedCityId,
            selectedBranchId,
            // حفظ التواريخ كما هي (strings بصيغة 'YYYY-MM-DD' أو ISO)
            selectedRange: selectedRange && selectedRange.length > 0 
              ? selectedRange.map(date => {
                  if (!date) return null;
                  // إذا كان التاريخ string، احفظه مباشرة
                  if (typeof date === 'string') return date;
                  // إذا كان dayjs object، حوله إلى string
                  if (date && typeof date === 'object' && date.format) {
                    return date.format('YYYY-MM-DD');
                  }
                  return null;
                })
              : null,
            startTime,
            selectedCarTypes,
            timestamp: new Date().toISOString()
          };
          sessionStorage.setItem('rental_branch_form', JSON.stringify(dataToSave));
        }
      } catch (error) {
        console.error('Error saving rental branch form data:', error);
      }
    }
  }, [selectedCityId, selectedBranchId, selectedRange, startTime, selectedCarTypes]);

  // استخدام hook مواعيد العمل
  const { workingHours, fetchWorkingHours, isLoading: workingHoursLoading } = useBranchWorkingHours();

  // جلب مواعيد العمل عند تغيير الفرع المحدد
  useEffect(() => {
    if (selectedBranchId) {
      fetchWorkingHours(selectedBranchId);
    }
  }, [selectedBranchId, fetchWorkingHours]);

  // التحقق من اكتمال جميع البيانات المطلوبة
  const isFormComplete = () => {
    return selectedCityId &&
      selectedBranchId &&
      selectedRange &&
      selectedRange.length === 2 &&
      selectedRange[0] &&
      selectedRange[1] &&
      startTime;
  };

  const showModal = () => {
    setOpen(true);
  };

  const handleSelectionChange = (typeId, isSelected) => {
    const currentSelectedCarTypes = selectedCarTypes || [];

    if (isSelected) {
      const newSelectedCarTypes = [...currentSelectedCarTypes, typeId];
      dispatch(setSelectedCarTypes(newSelectedCarTypes));
    } else {
      const newSelectedCarTypes = currentSelectedCarTypes.filter(id => id !== typeId);
      dispatch(setSelectedCarTypes(newSelectedCarTypes));
    }
  };

  const handleSearch = async () => {
    try {
      setIsLoading(true);
      
      // إذا لم يتم اختيار أي نوع، اختر جميع الأنواع
      if (!selectedCarTypes || selectedCarTypes.length === 0) {
        if (carTypes && carTypes.length > 0) {
          const allCarTypeIds = carTypes.map(carType => carType.typeId || carType.id);
          dispatch(setSelectedCarTypes(allCarTypeIds));
        }
      }
      
      await handleDataChange();
      // إبقاء الـ loading نشط حتى يتم الانتقال للصفحة التالية
      // سيتم إيقافه تلقائياً عند إعادة تحميل الصفحة أو الانتقال
    } catch (error) {
      // Handle error silently
      setIsLoading(false);
    }
  };

  const handleSelectAll = () => {
    try {
      if (carTypes && carTypes.length > 0) {
        if (isAllSelected) {
          // إذا كان الكل مختار، ألغي اختيار الكل
          dispatch(setSelectedCarTypes([]));
        } else {
          // إذا لم يكن الكل مختار، اختر الكل
          const allCarTypeIds = carTypes.map(carType => carType.typeId);
          dispatch(setSelectedCarTypes(allCarTypeIds));
        }
      }
    } catch (error) {
      // Handle error silently
    }
  };
  

  const isAllSelected = selectedCarTypes && carTypes && selectedCarTypes.length === carTypes.length;

  return (
    <>
      <div className="pt-4">

        <Row gutter={[20, 20]} align="top">
          <Col xl={4} lg={4} md={6} sm={12} xs={12}>
            <CitySelector
              value={selectedCityId}
              onChange={handleCityChange}
              placeholder={t('select_city')}
            />
          </Col>
          <Col xl={5} lg={5} md={6} sm={12} xs={12}>
            <BranchSelector
              selectedCityId={selectedCityId}
              value={selectedBranchId}
              onChange={handleBranchChange}
              placeholder={t('pickup_branch')}
            />
          </Col>
          <Col xl={8} lg={8} md={12} sm={24} xs={24}>
            <DateRangePicker
              locale={locale}
              workingHours={workingHours}
              selectedBranchId={selectedBranchId}
            />
          </Col>
          <Col xl={4} lg={4} md={6} sm={12} xs={12}>
            <StartTimePicker
              workingHours={workingHours}
              selectedBranchId={selectedBranchId}
            />
          </Col>
          <Col xl={3} lg={3} md={6} sm={12} xs={12}>
            <Button
              size="large"
              type="primary"
              className="!bg-[#162A2B] h-[50px]  !text-white px-4 md:px-10 w-full text-sm md:text-base disabled:!bg-gray-400 disabled:cursor-not-allowed max-sm:!h-[40px]   max-sm:!text-xs max-sm:!px-2 max-sm:!font-medium"
              onClick={showModal}
            // disabled={!isFormComplete()}
            >
              {t('next')}
            </Button>
          </Col>
        </Row>
      </div>

      <Modal
        title={<span className="text-lg md:text-xl font-bold text-center w-full block max-sm:!text-sm max-sm:!font-medium">{t('select_car_type_title')}</span>}
        open={open}
        onCancel={closeModal}
        footer={
          <div className="flex flex-col md:flex-row justify-center items-center gap-2 md:gap-4 pt-4 max-sm:gap-1">
            <Button
              type="primary"
              className='!bg-[#fff] !text-main shadow-none border border-main h-[40px] w-full md:w-[100px] text-sm md:text-base max-sm:!h-6 max-sm:!text-xs max-sm:!font-medium'
              icon={<CaretRightOutlined />}
              onClick={async () => {
                try {
                  setIsLoading(true);
                  
                  // إلغاء أي اختيارات سابقة للأنواع
                  dispatch(setSelectedCarTypes([]));
                  
                  // تخطي بدون اختيار أي أنواع سيارات - تمرير مصفوفة فارغة مباشرة
                  await handleDataChange(null, []); 
                } catch (error) {
                  // Handle error silently
                  setIsLoading(false);
                }
              }}>
              {t('skip')}
            </Button>
            <Button
              type="primary"
              className='shadow-none h-[40px] w-full md:w-[250px] text-sm md:text-base max-sm:!h-6 max-sm:!text-xs max-sm:!font-medium'
              icon={<SearchOutlined />}
              onClick={handleSearch}
              loading={isLoading}
              disabled={isLoading}
            >
              {isLoading ? t('searching') : t('search_for_results')}
            </Button>
          </div>
        }
      >
        <span className='text-center text-md block max-sm:!text-xs max-sm:!font-medium    '>{t('select_car_type')}</span>

        {/* Select All / Deselect All Checkboxes */}
        {carTypes && carTypes.length > 0 && (
          <div className="mt-4 p-2 border border-gray-200 rounded-lg bg-gray-50/50">
            <div className="flex items-center justify-between">
              <Checkbox
                checked={isAllSelected}
                onChange={handleSelectAll}
                className="!text-primary"
              >
                <span className="text-sm text-gray-700 font-medium">
                  {isAllSelected ? t('deselect_all') : t('select_all')}
                </span>
              </Checkbox>
              
              <span className="text-xs text-gray-500 bg-white px-2 py-1 rounded-full">
                {selectedCarTypes?.length || 0}/{carTypes.length}
              </span>
            </div>
          </div>
        )}

        <Row gutter={[12, 12]} className='mt-8 '>
          {carTypes && carTypes.length > 0 ? (
            carTypes.map((carType) => (
              <Col lg={12} md={12} sm={12} xs={12} key={carType.typeId || carType.id}>
                <CarTypeSelector
                  locale={locale}
                  data={carType}
                  onSelectionChange={(typeId, isSelected) => handleSelectionChange(typeId, isSelected)}
                />
              </Col>
            ))
          ) : (
            <Col span={24}>
              <div className="text-center text-gray-500 py-8 text-sm md:text-base max-sm:!text-xs max-sm:!font-medium">
                {t('no_car_types_available') || 'لا توجد أنواع سيارات متاحة'}
              </div>
            </Col>
          )}
        </Row>
      </Modal>
    </>
  );
};

export default RentalBranchForm;
