'use client';

import { CalendarOutlined, SearchOutlined } from '@ant-design/icons';
import { Button, Col, Form, Row } from 'antd';
import dayjs from 'dayjs';
import { useTranslations } from 'next-intl';
import { useDispatch, useSelector } from 'react-redux';

import useFormSessionStorage from '@/hook/useFormSessionStorage';
import { useRouter } from '@/navigation';
import cookieStorage from '@/utils/cookieStorage';
import File from '@components/svg/File';
import { CustomDatePicker, CustomDropdown, CustomInput, DeliveryLocation } from '@components/ui';
import { setReceptionData } from '@store/features/lim/limousineSlice';
import { useEffect, useRef, useState } from 'react';

const LimCarsRentHourDay = ({ locale, rentalType }) => {
    const [form] = Form.useForm();
    const [isSearching, setIsSearching] = useState(false);
    const dispatch = useDispatch();
    const router = useRouter();
    const t = useTranslations();
    const { settings } = useSelector((state) => state.shared);
    const searchTimeoutRef = useRef(null);

    // استخدام hook لحفظ واسترجاع البيانات من session
    const formKey = rentalType === 'hourly' ? 'lim_hourly_form' : 'lim_daily_form';
    const { saveToSession, getSavedData } = useFormSessionStorage(
        formKey,
        form,
        { rentalType }
    );

    // استرجاع البيانات المحفوظة عند التحميل
    useEffect(() => {
        const savedData = getSavedData();
        if (savedData && savedData.formValues) {
            const formValues = { ...savedData.formValues };
            // تحويل التاريخ من ISO string إلى dayjs object
            if (formValues.receptionDateTime) {
                formValues.receptionDateTime = dayjs(formValues.receptionDateTime);
            }
            form.setFieldsValue(formValues);
        }
    }, [rentalType]);

    // حفظ البيانات عند تغيير أي قيمة في النموذج
    useEffect(() => {
        const handleFormChange = () => {
            saveToSession({ rentalType });
        };

        // إضافة مستمع لتغيرات النموذج
        const interval = setInterval(() => {
            const values = form.getFieldsValue();
            if (values && Object.keys(values).length > 0) {
                handleFormChange();
            }
        }, 1000); // حفظ كل ثانية إذا كان هناك تغيير

        return () => clearInterval(interval);
    }, [rentalType, form, saveToSession]);

    // Cleanup timeout on unmount
    useEffect(() => {
        return () => {
            if (searchTimeoutRef.current) {
                clearTimeout(searchTimeoutRef.current);
            }
        };
    }, []);

    const onFinish = (values) => {
        try {
            // منع البحث إذا كان قيد التنفيذ
            if (isSearching) return;

            // بدء حالة التحميل
            setIsSearching(true);

            const payload = {
                ...values,
                typeLim: rentalType,
                receptionDateTime: values.receptionDateTime?.toISOString() || null,
                additionalHoursPerDay: values.additionalHoursPerDay || 0,
            };

            // حفظ البيانات
            cookieStorage.setItem('receptionData', JSON.stringify(payload));
            dispatch(setReceptionData(payload));

            const filtered = Object.fromEntries(
                Object.entries(payload).filter(([_, v]) => v != null)
            );
            const queryString = new URLSearchParams(filtered).toString();

            // إضافة تأخير للـ UX مع timeout للحماية
            searchTimeoutRef.current = setTimeout(() => {
                try {
                    // توجيه إلى نفس الصفحة مع البيانات
                    router.push(`/limo?${queryString}`);
                } catch (error) {
                    console.error('Error navigating to search results:', error);
                    setIsSearching(false);
                }
            }, 800); // تأخير 800ms لإظهار loading

            // إيقاف التحميل بعد 3 ثوان كحد أقصى
            setTimeout(() => {
                setIsSearching(false);
            }, 3000);

        } catch (error) {
            console.error('Error during search:', error);
            setIsSearching(false);
        }
    };

    const baseOptions = [
        { value: 'JEDDAH', label: t('limousine.jeddah') },
        { value: 'MAKKAH', label: t('limousine.makkah') },
    ];
    return (
        <Form form={form} onFinish={onFinish}>
            <Row gutter={[12, 0]} className="mt-4 md:mt-6">
                {/* المدينة */}
                <Col xs={12} sm={12} md={8} lg={4}>
                    <CustomDropdown
                        name="cityId"
                        options={baseOptions?.map(cityItem => ({
                            value: cityItem.value,
                            label: cityItem.label
                        })) || []}
                        placeholder={t('location.choose_city')}
                        className="w-full h-[45px] md:h-[50px]   max-sm:h-[40px] [&_.ant-select-selector]:max-sm:!text-xs  border border-[#162A2B] rounded text-sm"
                    />
                </Col>

                {/* العنوان */}
                <Col xs={12} sm={12} md={8} lg={6}>
                    <Form.Item
                        noStyle
                        shouldUpdate={(prevValues, currentValues) => prevValues.cityId !== currentValues.cityId}
                    >
                        {({ getFieldValue }) => {
                            const selectedCityId = getFieldValue('cityId');
                            return (
                                <DeliveryLocation
                                    form={form}
                                    addressName="addressMap"
                                    latName="latitude"
                                    lngName="longitude"
                                    label={t('location.choose_location')}
                                    className="text-xs truncate max-sm:!text-xs h-[50px] border border-[#162A2B] rounded-[3.5px] max-sm:!h-[40px]"
                                    skipApiValidation={true}
                                    cityId={selectedCityId}
                                    hideFormField={true}
                                />
                            );
                        }}
                    </Form.Item>
                </Col> 

                {/* عدد الساعات - يظهر فقط للحجز بالساعة */}
                {rentalType === 'hourly' && (
                    <Col xs={12} sm={12} md={8} lg={4}>
                        <CustomInput
                            name="hours"
                            suffix={<span className='text-textLight text-xs md:text-sm'>{t('location.hours_suffix')}</span>}
                            placeholder={t('location.hours_count')}
                            className="w-full h-[45px] md:h-[50px] !border !border-[#162A2B] !shadow-none rounded text-sm  max-sm:!h-[40px]  max-sm:!text-xs"
                            customRules={[
                                { required: true, message: t('required') },
                                {
                                    validator: (_, value) => {
                                        if (value && (value <= 0 || value === '0' || value === 0)) {
                                            return Promise.reject(new Error(t('limousine.validation_messages.hours_minimum')));
                                        }
                                        return Promise.resolve();
                                    }
                                },
                                {
                                    validator: (_, value) => {
                                        const maxHours = Number(settings?.["NUM_OF_ ADDITIONAL_HOURS_PER_DAY_FOR_LIMOUSINE"] - 1);
                                        // const maxHours = 9;
                                        if (value > maxHours) {
                                            return Promise.reject(new Error(t('limousine.validation_messages.hours_maximum', { maxHours: maxHours })));
                                        }
                                        return Promise.resolve();
                                    }
                                },
                                {
                                    validator: (_, value) => {
                                        if (value) {
                                            const arabicNumbers = /[\u0660-\u0669]/;
                                            if (arabicNumbers.test(value.toString())) {
                                                return Promise.reject(new Error(t('limousine.validation_messages.english_numbers_only')));
                                            }
                                        }
                                        return Promise.resolve();
                                    }
                                },
                                {
                                    validator: (_, value) => {
                                        if (value) {
                                            const decimalPattern = /\./;
                                            if (decimalPattern.test(value.toString())) {
                                                return Promise.reject(new Error(t('limousine.validation_messages.whole_numbers_only')));
                                            }
                                        }
                                        return Promise.resolve();
                                    }
                                }
                            ]}
                        />
                    </Col>
                )}

                {rentalType === 'daily' && (
                    <Col xs={12} sm={12} md={8} lg={4}>
                        <CustomInput
                            name="days"
                            suffix={<span className='text-textLight text-xs md:text-sm'>{t('location.days_suffix')}</span>}
                            placeholder={t('location.days_count')}
                            className="w-full h-[45px] md:h-[50px] !border !border-[#162A2B] !shadow-none rounded text-sm  max-sm:!h-[40px]  max-sm:!text-xs"
                            customRules={[
                                { required: true, message: t('required') },
                                {
                                    validator: (_, value) => {
                                        if (value && (value <= 0 || value === '0' || value === 0)) {
                                            return Promise.reject(new Error(t('required')));
                                        }
                                        return Promise.resolve();
                                    }
                                },
                                {
                                    validator: (_, value) => {
                                        const maxHours = 50;
                                        // const maxHours =9;
                                        if (value > maxHours) {
                                            return Promise.reject(new Error(t('limousine.validation_messages.days_maximum')));
                                        }
                                        return Promise.resolve();
                                    }
                                },
                                {
                                    validator: (_, value) => {
                                        if (value) {
                                            const arabicNumbers = /[\u0660-\u0669]/;
                                            if (arabicNumbers.test(value.toString())) {
                                                return Promise.reject(new Error(t('limousine.validation_messages.english_numbers_only')));
                                            }
                                        }
                                        return Promise.resolve();
                                    }
                                },

                                {
                                    validator: (_, value) => {
                                        if (value) {
                                            const decimalPattern = /\./;
                                            if (decimalPattern.test(value.toString())) {
                                                return Promise.reject(new Error(t('limousine.validation_messages.whole_numbers_only')));
                                            }
                                        }
                                        return Promise.resolve();
                                    }
                                }
                            ]}
                        />
                    </Col>
                )}

                {/* التاريخ والوقت */}
                <Col xs={12} sm={12} md={8} lg={6}>
                    <CustomDatePicker
                        name="receptionDateTime"
                        placeholder={t('location.date_time')}
                        size="large"
                        showTime
                        format="YYYY-MM-DD HH:mm"
                        className="w-full h-[45px] md:h-[50px] border border-[#162A2B] rounded text-sm max-sm:!h-[40px] [&_.ant-picker-input>input]:max-sm:!text-xs"
                        suffixIcon={<CalendarOutlined />}
                        maxDaysFromNow={settings?.["NUM_HOURS_FOR_START_RESERVATION_LIMOUSINE"]}
                    />
                </Col>

                {/* زر البحث */}
                <Col xs={24} sm={24} md={8} lg={4}>
                    <Button
                        size="large"
                        type="primary"
                        loading={isSearching}
                        disabled={isSearching}
                        className="w-full h-[50px] max-sm:!h-[40px] [&_>span]:max-sm:!text-sm   !bg-[#162A2B] !text-white flex items-center justify-center gap-1 min-w-[120px]"
                        onClick={() => form.submit()}
                    >
                        {!isSearching && <SearchOutlined />}
                        <span className="  sm:inline">
                            {isSearching ? t('search.searching') : t('search.search_results')}
                        </span>
                    </Button>
                </Col>
            </Row>
            <div className='flex gap-2 items-center text-[#A37C2D] mt-3 px-1'>
                <File />
                <span className="text-xs md:text-sm">{t('location.service_within_city_only')}</span>
            </div>

        </Form>
    );
};

export default LimCarsRentHourDay;