// components/AdvancedRentalPanel/forms/CompanyForm.jsx
'use client';

import CountryPhone from '@/app/(components)/ui/phone/CountryPhone';
import { URL } from '@/constants/api';
import { fetchSharedData } from '@store/shared/sharedSlice';
import { Button, Col, Form, Input, message, Row, Select } from 'antd';
import { useLocale, useTranslations } from 'next-intl';
import { useEffect, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';

const CompanyForm = () => {
  const locale = useLocale();
  const [form] = Form.useForm();
  const [loading, setLoading] = useState(false);
  const [isPhoneValid, setIsPhoneValid] = useState(false);
  const dispatch = useDispatch();
  const t = useTranslations('company_form');
  // الحصول على البيانات من Redux store
  const { carBrands, loading: sharedLoading } = useSelector((state) => state.shared);
  const { user } = useSelector((state) => state.profile);

  // جلب البيانات المشتركة عند تحميل المكون
  useEffect(() => {
    if (!carBrands || carBrands.length === 0) {
      dispatch(fetchSharedData());
    }
  }, [dispatch, carBrands]);
  const capitalize = (str) => {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
  };
  // ملء حقول النموذج ببيانات المستخدم عند تسجيل الدخول
  useEffect(() => {
    if (user) {
      form.setFieldsValue({
        responsiblePerson: capitalize(user?.firstName) + ' ' + capitalize(user?.lastName),
        phoneNumber: user.phone || user.mobile || user.phoneNumber,
        email: user.email
      });
    }
  }, [user, form]);

  // مراقبة تغييرات form لضمان تحديث CountryPhone
  const phoneValue = Form.useWatch('phoneNumber', form);

  const onFinish = async (values) => {
    setLoading(true);
    try {
      // تحويل البيانات لتتطابق مع API
      const apiData = {
        companyName: values.companyName,
        activity: values.activity,
        city: values.city,
        responsibleName: values.responsiblePerson,
        mobile: values.phoneNumber,
        email: values.email,
        carsNumber: parseInt(values.carCount),
        months: parseInt(values.rentalDuration),
        brandIds: Array.isArray(values.carBrands) ? values.carBrands.map(id => parseInt(id)) : []
      };


      const apiUrl = URL('/company-requests');

      if (!apiUrl) {
        throw new Error('API URL غير محدد');
      }

      // إرسال البيانات للـ API
      const response = await fetch(apiUrl, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify(apiData)
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('Error response:', errorText);

        let errorMessage = 'فشل في إرسال الطلب';
        try {
          const errorResult = JSON.parse(errorText);
          errorMessage = errorResult.message || errorMessage;
        } catch (e) {
          // إذا فشل في parse JSON، استخدم النص كما هو
          if (errorText) {
            errorMessage = errorText;
          }
        }

        throw new Error(errorMessage);
      }

      const result = await response.json();

      message.success(t('messages.success'));

      // الاحتفاظ ببيانات المستخدم ومسح باقي الحقول فقط
      if (user) {
        form.setFieldsValue({
          responsiblePerson: capitalize(user?.firstName) + ' ' + capitalize(user?.lastName),
          phoneNumber: user.phone || user.mobile || user.phoneNumber,
          email: user.email,
          // مسح باقي الحقول
          carCount: undefined,
          rentalDuration: undefined,
          carBrands: undefined,
          companyName: undefined,
          city: undefined,
          activity: undefined
        });
      } else {
        form.resetFields();
      }
    } catch (error) {
      console.error('API Error:', error);

      // معالجة خاصة لأخطاء CORS
      if (error.message.includes('CORS') || error.message.includes('Failed to fetch')) {
        message.error(t('messages.connection_error'));
      } else {
        message.error(error.message || t('messages.error'));
      }
    } finally {
      setLoading(false);
    }
  };

  const onFinishFailed = (errorInfo) => {
    message.error(t('messages.validation_error'));
  };

  return ( 
      <Form
        form={form}
        name="companyForm"
        onFinish={onFinish}
        onFinishFailed={onFinishFailed}
        layout="vertical"
        autoComplete="off"
      >
        <Row gutter={[16, 0]} className="[&_.ant-form-item-required]:max-sm:!text-xs">
          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="carCount"
              label={t('car_count')}
              required
              rules={[
                {
                  validator: (_, value) => {
                    // إذا كان الحقل فارغ أو undefined
                    if (!value || value === '' || value === null || value === undefined) {
                      return Promise.reject(new Error(t('validation.car_count_required')));
                    }

                    const numValue = parseInt(value);
                    if (isNaN(numValue) || numValue < 1) {
                      return Promise.reject(new Error(t('validation.car_count_greater_zero')));
                    }
                    return Promise.resolve();
                  }
                }
              ]}
            >
              <Input
                className="w-full h-[50px] border border-main rounded text-sm max-sm:!h-[40px] max-sm:!text-xs"
                placeholder={t('placeholders.car_count')}
                size="large"
                type="number"
              />
            </Form.Item>
          </Col>

          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="rentalDuration"
              label={t('rental_duration')}
              required
              rules={[
                { 
                  validator: (_, value) => {
                    // إذا كان الحقل فارغ أو undefined
                    if (!value || value === '' || value === null || value === undefined) {
                      return Promise.reject(new Error(t('validation.rental_duration_required')));
                    }

                    const numValue = parseInt(value);
                    if (isNaN(numValue) || numValue < 1) {
                      return Promise.reject(new Error(t('validation.car_count_greater_zero')));
                    }
                    return Promise.resolve();
                  }
                },
              ]}
            >
              <Input
                className="w-full h-[50px] border border-main rounded text-sm max-sm:!h-[40px] max-sm:!text-xs"
                placeholder={t('placeholders.rental_duration')}
                size="large"
                type="number"
              />
            </Form.Item>
          </Col>

          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="carBrands"
              label={t('car_brands')}
              required
              rules={[
                { required: true, message: t('validation.car_brands_required') }
              ]}
              className="[&_.ant-select-selection-overflow]:!mt-[4px] "
            >

              <Select
                mode="multiple"
                placeholder={t('placeholders.car_brands')}
                className="w-full  !h-[50px]  text-sm max-sm:!h-[40px] max-sm:!text-xs border border-main rounded"
                size="large"
                loading={sharedLoading}
                maxTagCount="responsive"
                maxTagTextLength={10}
                showSearch
                filterOption={(input, option) =>
                  (option?.label ?? '').toLowerCase().includes(input.toLowerCase())
                }
                tagRender={(props) => (
                  <span className="  px-2 py-0 rounded-full text-xs font-medium bg-main text-white mr-2 mb-1">
                    {props.label}
                    <button
                      type="button"
                      className="    w-4 h-4 rounded-full text-white hover:bg-white hover:text-main transition-colors"
                      onClick={props.onClose}
                    >
                      ×
                    </button>
                  </span>
                )}
                options={carBrands?.map(brand => ({
                  value: brand.brandId,
                  label: locale === 'en' ? brand.brandName : brand.arabicName
                })) || []}
              />
            </Form.Item>
          </Col>
          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="companyName"
              label={t('company_name')}
              required
              rules={[
                { required: true, message: t('validation.company_name_required') },
                { min: 3, message: t('validation.company_name_min') }
              ]}
            >
              <Input
                className="w-full h-[50px] border border-main rounded text-sm max-sm:!h-[40px] max-sm:!text-xs"
                placeholder={t('placeholders.company_name')}
                size="large"
              />
            </Form.Item>
          </Col>

          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="city"
              label={t('city')}
              required
              rules={[
                { required: true, message: t('validation.city_required') }
              ]}
            >
              <Input
                className="w-full h-[50px] border border-main rounded text-sm max-sm:!h-[40px] max-sm:!text-xs"
                placeholder={t('placeholders.city')}
                size="large"
              />
            </Form.Item>
          </Col>
          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="activity"
              label={t('activity')}
              required
              rules={[
                { required: true, message: t('validation.activity_required') },
                { min: 3, message: t('validation.activity_min') }
              ]}
            >
              <Input
                className="w-full h-[50px] border border-main rounded text-sm max-sm:!h-[40px] max-sm:!text-xs"
                placeholder={t('placeholders.activity')}
                size="large"
              />
            </Form.Item>
          </Col>

          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="responsiblePerson"
              label={t('responsible_person')}
              required
              rules={[
                { required: true, message: t('validation.responsible_person_required') },
                { min: 3, message: t('validation.responsible_person_min') }
              ]}
            >
              <Input
                className="w-full h-[50px] border border-main rounded text-sm max-sm:!h-[40px] max-sm:!text-xs"
                placeholder={t('placeholders.responsible_person')}
                size="large"
              />
            </Form.Item>
          </Col>

          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="phoneNumber"
              label={t('phone_number')}
              required
              rules={[
                { required: true, message: t('validation.phone_required') },
              ]}
            >
              <div dir="ltr">
                <CountryPhone
                  placeholder={t('placeholders.phone_number')}
                  defaultCountry="sa"
                  value={phoneValue}
                  className="max-sm:!h-[40px] max-sm:!text-xs"
                  onChange={(phone) => {
                    // Remove leading zero if user types it
                    if (phone && phone.startsWith('0')) {
                      const phoneWithoutZero = phone.substring(1);
                      form.setFieldsValue({ phoneNumber: phoneWithoutZero });
                    } else {
                      form.setFieldsValue({ phoneNumber: phone });
                    }
                  }}
                  onValidationChange={setIsPhoneValid}
                />
              </div>
            </Form.Item>
          </Col>

          <Col md={8} lg={6} sm={12} xs={12}>
            <Form.Item
              name="email"
              label={t('email')}
              required
              rules={[
                { required: true, message: t('validation.email_required') },
                { type: 'email', message: t('validation.email_valid') }
              ]}
            >
              <Input
                className="w-full h-[50px] border border-main rounded text-sm max-sm:!h-[40px] max-sm:!text-xs"
                placeholder={t('placeholders.email')}
                size="large"
              />
            </Form.Item>
          </Col>
          <Col md={8} lg={6} sm={12} xs={12} className='flex  items-center'>
            {/* <Form.Item> */}
            <Button
              size="large"
              type="primary"
              htmlType="submit"
              loading={loading}
              className="!bg-main !text-white px-10  w-full h-[50px] block max-sm:!h-[40px] max-sm:!text-xs max-sm:!font-medium"
            >
              {loading ? t('submitting') : t('submit_button')}
            </Button>
            {/* </Form.Item> */}
          </Col>


        </Row>

      </Form>
  );
};

export default CompanyForm;
